/*
 * Copyright 2013, Google Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jf.dexlib2.dexbacked.raw;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.jf.dexlib2.dexbacked.DexBackedDexFile;
import org.jf.dexlib2.dexbacked.raw.util.DexAnnotator;
import org.jf.dexlib2.util.AnnotatedBytes;

public class TypeListItem {
    public static final int SIZE_OFFSET = 0;
    public static final int LIST_OFFSET = 4;

    @NonNull
    public static SectionAnnotator makeAnnotator(@NonNull DexAnnotator annotator, @NonNull MapItem mapItem) {
        return new SectionAnnotator(annotator, mapItem) {
            @NonNull
            @Override
            public String getItemName() {
                return "type_list";
            }

            @Override
            protected void annotateItem(@NonNull AnnotatedBytes out, int itemIndex, @Nullable String itemIdentity) {
                int size = dexFile.readSmallUint(out.getCursor());
                out.annotate(4, "size: %d", size);

                for (int i = 0; i < size; i++) {
                    int typeIndex = dexFile.readUshort(out.getCursor());
                    out.annotate(2, TypeIdItem.getReferenceAnnotation(dexFile, typeIndex));
                }
            }

            @Override
            public int getItemAlignment() {
                return 4;
            }
        };
    }

    @NonNull
    public static String getReferenceAnnotation(@NonNull DexBackedDexFile dexFile, int typeListOffset) {
        if (typeListOffset == 0) {
            return "type_list_item[NO_OFFSET]";
        }

        try {
            String typeList = asString(dexFile, typeListOffset);
            return String.format("type_list_item[0x%x]: %s", typeListOffset, typeList);
        } catch (Exception ex) {
            ex.printStackTrace(System.err);
        }
        return String.format("type_list_item[0x%x]", typeListOffset);
    }

    @NonNull
    public static String asString(@NonNull DexBackedDexFile dexFile, int typeListOffset) {
        if (typeListOffset == 0) {
            return "";
        }

        StringBuilder sb = new StringBuilder();

        int size = dexFile.readSmallUint(typeListOffset);
        for (int i = 0; i < size; i++) {
            int typeIndex = dexFile.readUshort(typeListOffset + 4 + i * 2);
            String type = dexFile.getType(typeIndex);
            sb.append(type);
        }
        return sb.toString();
    }
}
