<?php

namespace EAddonsPdf\Modules\Pdf\Widgets;

use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Icons_Manager;
use Elementor\Controls_Manager;
use EAddonsForElementor\Base\Base_Widget;
use EAddonsForElementor\Core\Utils;

/**
 * PDF Button
 *
 * Elementor widget for e-addons
 * Extend the native "Button" Widget
 * /elementor/includes/widgets/button.php
 */
class Button extends Base_Widget {

    use \EAddonsPdf\Modules\Pdf\Traits\Pdf;
    //use \Elementor\Includes\Widgets\Traits\Button;

    public function get_name() {
        return 'button_pdf';
    }

    public function get_title() {
        return esc_html__('Button PDF Download', 'e-addons');
    }

    public function get_icon() {
        return 'eadd-button-pdf';
    }

    public function get_pid() {
        return 223;
    }

    /**
     * Register button widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     * 
     * Extend the base Button
     * /elementor/includes/widgets/button.php
     */
    protected function register_controls() {
        $this->start_controls_section(
                'section_button',
                [
                    'label' => esc_html__('Button', 'elementor'),
                ]
        );

        $this->add_control(
                'button_type',
                [
                    'label' => esc_html__('Type', 'elementor'),
                    'type' => Controls_Manager::SELECT,
                    'default' => '',
                    'options' => [
                        '' => esc_html__('Default', 'elementor'),
                        'info' => esc_html__('Info', 'elementor'),
                        'success' => esc_html__('Success', 'elementor'),
                        'warning' => esc_html__('Warning', 'elementor'),
                        'danger' => esc_html__('Danger', 'elementor'),
                    ],
                    'prefix_class' => 'elementor-button-',
                ]
        );

        $this->add_control(
                'text',
                [
                    'label' => esc_html__('Text', 'elementor'),
                    'type' => Controls_Manager::TEXT,
                    'dynamic' => [
                        'active' => true,
                    ],
                    'default' => esc_html__('Click here to Download PDF', 'elementor'),
                    'placeholder' => esc_html__('Click here to Download PDF', 'elementor'),
                ]
        );

        $this->add_control(
                'nofollow',
                [
                    'label' => esc_html__('Link nofollow', 'elementor'),
                    'type' => Controls_Manager::SWITCHER,
                ]
        );

        $this->add_responsive_control(
                'align',
                [
                    'label' => esc_html__('Alignment', 'elementor'),
                    'type' => Controls_Manager::CHOOSE,
                    'options' => [
                        'left' => [
                            'title' => esc_html__('Left', 'elementor'),
                            'icon' => 'eicon-text-align-left',
                        ],
                        'center' => [
                            'title' => esc_html__('Center', 'elementor'),
                            'icon' => 'eicon-text-align-center',
                        ],
                        'right' => [
                            'title' => esc_html__('Right', 'elementor'),
                            'icon' => 'eicon-text-align-right',
                        ],
                        'justify' => [
                            'title' => esc_html__('Justified', 'elementor'),
                            'icon' => 'eicon-text-align-justify',
                        ],
                    ],
                    'prefix_class' => 'elementor%s-align-',
                    'default' => '',
                ]
        );

        $this->add_control(
                'size',
                [
                    'label' => esc_html__('Size', 'elementor'),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'sm',
                    'options' => self::get_button_sizes(),
                    'style_transfer' => true,
                ]
        );

        $this->add_control(
                'selected_icon',
                [
                    'label' => esc_html__('Icon', 'elementor'),
                    'type' => Controls_Manager::ICONS,
                    'fa4compatibility' => 'icon',
                    'skin' => 'inline',
                    'label_block' => false,
                ]
        );

        $this->add_control(
                'icon_align',
                [
                    'label' => esc_html__('Icon Position', 'elementor'),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'left',
                    'options' => [
                        'left' => esc_html__('Before', 'elementor'),
                        'right' => esc_html__('After', 'elementor'),
                    ],
                    'condition' => [
                        'selected_icon[value]!' => '',
                    ],
                ]
        );

        $this->add_control(
                'icon_indent',
                [
                    'label' => esc_html__('Icon Spacing', 'elementor'),
                    'type' => Controls_Manager::SLIDER,
                    'range' => [
                        'px' => [
                            'max' => 50,
                        ],
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .elementor-button .elementor-align-icon-right' => 'margin-left: {{SIZE}}{{UNIT}};',
                        '{{WRAPPER}} .elementor-button .elementor-align-icon-left' => 'margin-right: {{SIZE}}{{UNIT}};',
                    ],
                ]
        );

        $this->add_control(
                'button_css_id',
                [
                    'label' => esc_html__('Button ID', 'elementor'),
                    'type' => Controls_Manager::TEXT,
                    'dynamic' => [
                        'active' => true,
                    ],
                    'default' => '',
                    'title' => esc_html__('Add your custom id WITHOUT the Pound key. e.g: my-id', 'elementor'),
                    'description' => sprintf(
                            /* translators: 1: Code open tag, 2: Code close tag. */
                            esc_html__('Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows %1$sA-z 0-9%2$s & underscore chars without spaces.', 'elementor'),
                            '<code>',
                            '</code>'
                    ),
                    'separator' => 'before',
                ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
                'section_style',
                [
                    'label' => esc_html__('Button', 'elementor'),
                    'tab' => Controls_Manager::TAB_STYLE,
                ]
        );
        
        //$this->register_button_content_controls();

        $this->add_group_control(
                Group_Control_Typography::get_type(),
                [
                    'name' => 'typography',
                    'global' => [
                        'default' => Global_Typography::TYPOGRAPHY_ACCENT,
                    ],
                    'selector' => '{{WRAPPER}} .elementor-button',
                ]
        );

        $this->add_group_control(
                Group_Control_Text_Shadow::get_type(),
                [
                    'name' => 'text_shadow',
                    'selector' => '{{WRAPPER}} .elementor-button',
                ]
        );

        $this->start_controls_tabs('tabs_button_style');

        $this->start_controls_tab(
                'tab_button_normal',
                [
                    'label' => esc_html__('Normal', 'elementor'),
                ]
        );

        $this->add_control(
                'button_text_color',
                [
                    'label' => esc_html__('Text Color', 'elementor'),
                    'type' => Controls_Manager::COLOR,
                    'default' => '',
                    'selectors' => [
                        '{{WRAPPER}} .elementor-button' => 'fill: {{VALUE}}; color: {{VALUE}};',
                    ],
                ]
        );

        $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'background',
                    'label' => esc_html__('Background', 'elementor'),
                    'types' => ['classic', 'gradient'],
                    'exclude' => ['image'],
                    'selector' => '{{WRAPPER}} .elementor-button',
                    'fields_options' => [
                        'background' => [
                            'default' => 'classic',
                        ],
                        'color' => [
                            'global' => [
                                'default' => Global_Colors::COLOR_ACCENT,
                            ],
                        ],
                    ],
                ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
                'tab_button_hover',
                [
                    'label' => esc_html__('Hover', 'elementor'),
                ]
        );

        $this->add_control(
                'hover_color',
                [
                    'label' => esc_html__('Text Color', 'elementor'),
                    'type' => Controls_Manager::COLOR,
                    'selectors' => [
                        '{{WRAPPER}} .elementor-button:hover, {{WRAPPER}} .elementor-button:focus' => 'color: {{VALUE}};',
                        '{{WRAPPER}} .elementor-button:hover svg, {{WRAPPER}} .elementor-button:focus svg' => 'fill: {{VALUE}};',
                    ],
                ]
        );

        $this->add_group_control(
                Group_Control_Background::get_type(),
                [
                    'name' => 'button_background_hover',
                    'label' => esc_html__('Background', 'elementor'),
                    'types' => ['classic', 'gradient'],
                    'exclude' => ['image'],
                    'selector' => '{{WRAPPER}} .elementor-button:hover, {{WRAPPER}} .elementor-button:focus',
                    'fields_options' => [
                        'background' => [
                            'default' => 'classic',
                        ],
                    ],
                ]
        );

        $this->add_control(
                'button_hover_border_color',
                [
                    'label' => esc_html__('Border Color', 'elementor'),
                    'type' => Controls_Manager::COLOR,
                    'condition' => [
                        'border_border!' => '',
                    ],
                    'selectors' => [
                        '{{WRAPPER}} .elementor-button:hover, {{WRAPPER}} .elementor-button:focus' => 'border-color: {{VALUE}};',
                    ],
                ]
        );

        $this->add_control(
                'hover_animation',
                [
                    'label' => esc_html__('Hover Animation', 'elementor'),
                    'type' => Controls_Manager::HOVER_ANIMATION,
                ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_group_control(
                Group_Control_Border::get_type(),
                [
                    'name' => 'border',
                    'selector' => '{{WRAPPER}} .elementor-button',
                    'separator' => 'before',
                ]
        );

        $this->add_control(
                'border_radius',
                [
                    'label' => esc_html__('Border Radius', 'elementor'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', '%', 'em'],
                    'selectors' => [
                        '{{WRAPPER}} .elementor-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                ]
        );

        $this->add_group_control(
                Group_Control_Box_Shadow::get_type(),
                [
                    'name' => 'button_box_shadow',
                    'selector' => '{{WRAPPER}} .elementor-button',
                ]
        );

        $this->add_responsive_control(
                'text_padding',
                [
                    'label' => esc_html__('Padding', 'elementor'),
                    'type' => Controls_Manager::DIMENSIONS,
                    'size_units' => ['px', 'em', '%'],
                    'selectors' => [
                        '{{WRAPPER}} .elementor-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    ],
                    'separator' => 'before',
                ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
                'section_e_pdf',
                [
                    'label' => esc_html__('PDF', 'e-addons'),
                ]
        );

        $this->add_control(
                'e_pdf_button_converter',
                [
                    'label' => esc_html__('HTML to PDF Converter', 'e-addons'),
                    'type' => Controls_Manager::SELECT,
                    'options' => [
                        'mpdf' => esc_html__('mPDF', 'e-addons'),
                        'browser' => esc_html__('Browser', 'e-addons'),
                    ],
                    'default' => 'mpdf',
                ]
        );

        $this->add_control(
                'e_pdf_button_title', [
            'label' => esc_html__('File name', 'e-addons'),
            'label_block' => true,
            'type' => Controls_Manager::TEXT,
            'default' => '{{post.post_name}}',
            'description' => esc_html__('Write here the PDF file title without extension.', 'e-addons'),
                ]
        );

        $this->add_control(
                'e_pdf_h_header',
                [
                    'label' => esc_html__('Header', 'e-addons'),
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
        );
        $this->add_control(
                'e_pdf_header',
                [
                    'label' => esc_html__('Add Header', 'e-addons'),
                    'type' => Controls_Manager::SWITCHER,
                    'description' => 'Set an HTML page header.',
                ]
        );
        $this->add_control(
                'e_pdf_header_txt',
                [
                    //'label' => esc_html__('Header', 'e-addons'),
                    'type' => Controls_Manager::WYSIWYG,
                    'condition' => [
                        'e_pdf_header!' => '',
                    ],
                ]
        );
        $this->add_control(
                'e_pdf_header_aliases',
                [
                    'label' => '<b>' . esc_html__('Replaceable Aliases', 'e-addons') . '</b>',
                    'type' => Controls_Manager::RAW_HTML,
                    'raw' => 'There are several placemarkers you can include, which will be replaced when the PDF file is ouput: <b>{nb}</b> for total pages number, and <b>{PAGENO}</b> for current page number.' .
                    '<br><a href="https://mpdf.github.io/what-else-can-i-do/replaceable-aliases.html" target="_blank">Read more</a>',
                    'condition' => [
                        'e_pdf_header!' => '',
                    ],
                ]
        );
        $this->add_control(
                'e_pdf_header_margin', [
            'label' => esc_html__('Header Margin', 'e-addons'),
            'type' => Controls_Manager::NUMBER,
            'condition' => [
                'e_pdf_header!' => '',
            ],
                ]
        );

        $this->add_control(
                'e_pdf_button_body',
                [
                    'label' => esc_html__('Body', 'e-addons'),
                    'type' => Controls_Manager::SELECT,
                    'options' => [
                        'post' => esc_html__('Current Page', 'e-addons'),
                        'template' => esc_html__('Custom Template', 'e-addons'),
                    ],
                    'separator' => 'before',
                    'default' => 'post',
                ]
        );

        $this->add_control(
                'e_pdf_button_template',
                [
                    'label' => esc_html__('Template', 'e-addons'),
                    'type' => 'e-query',
                    'placeholder' => esc_html__('Template Name', 'e-addons'),
                    'label_block' => true,
                    'query_type' => 'posts',
                    'object_type' => 'elementor_library',
                    'description' => 'Use a Elementor Template as body fo this PDF.',
                    'condition' => [
                        'e_pdf_button_body' => 'template',
                    ],
                ]
        );

        $this->add_control(
                'e_pdf_h_footer',
                [
                    'label' => esc_html__('Footer', 'e-addons'),
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
        );
        $this->add_control(
                'e_pdf_footer',
                [
                    'label' => esc_html__('Add Footer', 'e-addons'),
                    'type' => Controls_Manager::SWITCHER,
                    'description' => 'Set an HTML page footer.',
                ]
        );
        $this->add_control(
                'e_pdf_footer_txt',
                [
                    //'label' => esc_html__('Footer', 'e-addons'),
                    'type' => Controls_Manager::WYSIWYG,
                    'condition' => [
                        'e_pdf_footer!' => '',
                    ],
                ]
        );
        $this->add_control(
                'e_pdf_footer_aliases',
                [
                    'label' => '<b>' . esc_html__('Replaceable Aliases', 'e-addons') . '</b>',
                    'type' => Controls_Manager::RAW_HTML,
                    'raw' => 'There are several placemarkers you can include, which will be replaced when the PDF file is ouput: <b>{nb}</b> for total pages number, and <b>{PAGENO}</b> for current page number.' .
                    '<br><a href="https://mpdf.github.io/what-else-can-i-do/replaceable-aliases.html" target="_blank">Read more</a>',
                    'condition' => [
                        'e_pdf_footer!' => '',
                    ],
                ]
        );
        $this->add_control(
                'e_pdf_footer_margin', [
            'label' => esc_html__('Footer Margin', 'e-addons'),
            'type' => Controls_Manager::NUMBER,
            'condition' => [
                'e_pdf_footer!' => '',
            ],
                ]
        );

        $this->add_control(
                'e_pdf_button_styles',
                [
                    'label' => esc_html__('Add Styles', 'e-addons'),
                    'type' => Controls_Manager::SELECT,
                    'options' => [
                        'unstyled' => esc_html__('No Style', 'e-addons'),
                        'elementor' => esc_html__('Elementor', 'e-addons'),
                        'all' => esc_html__('Elementor & Theme', 'e-addons'),
                    ],
                    'separator' => 'before',
                    'default' => 'elementor',
                ]
        );

        $this->add_control(
                'e_pdf_button_page',
                [
                    'label' => esc_html__('Page', 'e-addons'),
                    'type' => Controls_Manager::HEADING,
                    'separator' => 'before',
                ]
        );

        $this->add_control(
                'e_pdf_button_size',
                [
                    'label' => esc_html__('Size', 'e-addons'),
                    'type' => Controls_Manager::SELECT,
                    'default' => 'a4',
                    'options' => \EAddonsPdf\Modules\Pdf\Pdf::get_paper_sizes(),
                ]
        );

        $this->add_common_pdf_controls();

        $this->add_control(
                'e_pdf_section_page', [
            'label' => esc_html__('Sections as Pages', 'e-addons'),
            'type' => Controls_Manager::SWITCHER,
            'description' => esc_html__('Force every Template Section as a new page', 'e-addons'),
                ]
        );

        $this->add_control(
                'e_pdf_dpi',
                [
                    'label' => esc_html__('DPI', 'e-addons'),
                    'type' => Controls_Manager::SELECT,
                    'options' => [
                        '' => esc_html__('Default', 'e-addons'),
                        '72' => esc_html__('72', 'e-addons'),
                        '150' => esc_html__('150', 'e-addons'),
                        '300' => esc_html__('300', 'e-addons'),
                    ],
                    'condition' => [
                        'e_pdf_button_converter!' => 'browser',
                    ],
                ]
        );

        $this->add_bg_controls();

        $this->add_control(
                'e_pdf_uploads', [
            'label' => esc_html__('Use Uploads folder', 'e-addons'),
            'type' => Controls_Manager::SWITCHER,
            'separator' => 'before',
            'description' => esc_html__('Move the pdf file generator into the wp Uploads folder for security', 'e-addons'),
                ]
        );

        $this->end_controls_section();
    }

    /**
     * Render button widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        $pdf_url = $this->get_module_url() . 'front/pdf.php';

        if ($settings['e_pdf_uploads']) {
            // download pdf copy        
            //echo $this->get_plugin_path().'e-addons-pdf.php';
            //$plugin = get_plugin_data($this->get_plugin_path().'e-addons-pdf.php');
            //var_dump($plugin['Version']);
            $cloned = false;
            $tempDir = WP_CONTENT_DIR . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'mpdf';
            //$fileDir = $tempDir.DIRECTORY_SEPARATOR.$plugin['Version'];
            if (file_exists($tempDir . DIRECTORY_SEPARATOR . 'pdf.php')) {
                $cloned = true;
            } else {
                if (!is_dir($tempDir)) {
                    wp_mkdir_p($tempDir);
                }
                $dest = $this->get_module_path() . 'front' . DIRECTORY_SEPARATOR . 'pdf.php';
                //$cloned = copy($dest, $fileDir.DIRECTORY_SEPARATOR.'pdf.php');                
                file_put_contents($tempDir . DIRECTORY_SEPARATOR . 'pdf.php', '<?php include_once("' . $dest . '");');
                chmod($tempDir . DIRECTORY_SEPARATOR . 'pdf.php', 0755);
            }
            if ($cloned) {
                $dirs = wp_get_upload_dir();
                //var_dump($dirs);
                $pdf_url = $dirs['baseurl'] . '/mpdf/pdf.php';
            }
        }

        $pdf_url .= '?post_id=' . get_the_ID();
        $pdf_url .= '&element_id=' . $this->get_id();

        $this->add_render_attribute('wrapper', 'class', 'elementor-button-wrapper');

        $this->add_render_attribute('button', 'href', $pdf_url);
        $this->add_render_attribute('button', 'class', 'elementor-button-link');

        if ($settings['nofollow']) {
            $this->add_render_attribute('button', 'rel', 'nofollow');
        }

        $this->add_render_attribute('button', 'class', 'elementor-button');
        $this->add_render_attribute('button', 'role', 'button');

        if (!empty($settings['button_css_id'])) {
            $this->add_render_attribute('button', 'id', $settings['button_css_id']);
        }

        if (!empty($settings['size'])) {
            $this->add_render_attribute('button', 'class', 'elementor-size-' . $settings['size']);
        }

        if ($settings['hover_animation']) {
            $this->add_render_attribute('button', 'class', 'elementor-animation-' . $settings['hover_animation']);
        }
        ?>
        <div <?php echo $this->get_render_attribute_string('wrapper'); ?>>
            <a <?php echo $this->get_render_attribute_string('button'); ?>>
                <?php $this->render_text(); ?>
            </a>
        </div>
        <?php
    }

    /**
     * Get button sizes.
     *
     * Retrieve an array of button sizes for the button widget.
     *
     * @since 1.0.0
     * @access public
     * @static
     *
     * @return array An array containing button sizes.
     */
    public static function get_button_sizes() {
        return [
            'xs' => esc_html__('Extra Small', 'elementor'),
            'sm' => esc_html__('Small', 'elementor'),
            'md' => esc_html__('Medium', 'elementor'),
            'lg' => esc_html__('Large', 'elementor'),
            'xl' => esc_html__('Extra Large', 'elementor'),
        ];
    }

    /**
     * Render button text.
     *
     * Render button widget text.
     *
     * @since 1.5.0
     * @access protected
     */
    protected function render_text() {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute([
            'content-wrapper' => [
                'class' => ['elementor-button-content-wrapper', 'e-flex'],
            ],
            'icon-align' => [
                'class' => [
                    'elementor-button-icon',
                    'elementor-align-icon-' . $settings['icon_align'],
                ],
            ],
            'text' => [
                'class' => 'elementor-button-text',
            ],
        ]);

        $this->add_inline_editing_attributes('text', 'none');
        ?>
        <span <?php echo $this->get_render_attribute_string('content-wrapper'); ?>>
            <?php if (!empty($settings['icon']) || !empty($settings['selected_icon']['value'])) : ?>
                <span <?php echo $this->get_render_attribute_string('icon-align'); ?>>
                    <?php
                    Icons_Manager::render_icon($settings['selected_icon'], ['aria-hidden' => 'true']);
                    ?>
                </span>
            <?php endif; ?>
            <span <?php echo $this->get_render_attribute_string('text'); ?>><?php echo $settings['text']; ?></span>
        </span>
        <?php
    }

    public function on_import($element) {
        return Icons_Manager::on_import_migration($element, 'icon', 'selected_icon');
    }

}
