<?php

namespace EDD\Pro\Admin\Extensions;

use EDD\Admin\Extensions\ProductData;

class ExtensionPage extends \EDD\Admin\Extensions\ExtensionPage {
	use \EDD\Admin\Extensions\Traits\Buttons;

	/**
	 * Whether the current active pass can install an extension.
	 * We assume true by default.
	 *
	 * @var bool
	 */
	private $can_install = true;

	/**
	 * Outputs the cards.
	 *
	 * @since 3.1.1
	 * @return void
	 */
	protected function do_cards() {

		// If this is an inactive pro install, we will show all extensions without the break in between.
		if ( edd_is_inactive_pro() ) {
			$this->can_install = false;
			$settings_url      = edd_get_admin_url(
				array(
					'page' => 'edd-settings',
				)
			);

			$purchase_url = edd_link_helper(
				'https://easydigitaldownloads.com/pricing',
				array(
					'utm_medium'  => 'extensions-page',
					'utm_content' => 'missing-key-purchase',
				)
			);
			?>
			<div class="edd-extension-manager__key-notice">
				<p><?php esc_html_e( 'You have not yet added a license key. A valid license key is required in order to use our extensions.', 'easy-digital-downloads' ); ?></p>
				<div class="edd-extension-manager__key-actions">
					<a href="<?php echo esc_url( $settings_url ); ?>" class="button button-primary edd-extension-manager__action-settings"><?php esc_html_e( 'Enter License Key', 'easy-digital-downloads' ); ?></a>
					<a href="<?php echo $purchase_url; ?>" class="button edd-extension-manager__action-upgrade" target="_blank"><?php esc_html_e( 'Purchase License', 'easy-digital-downloads' ); ?></a>
				</div>
			</div>
			<?php
			parent::do_cards();
			return;
		}

		// At this point, we know the user can install at least some extensions.
		?>
		<div class="edd-extension-manager__card-group">
			<?php
			foreach ( $this->get_product_data() as $item_id => $extension ) {
				$this->maybe_insert_break( $extension );
				$this->do_single_extension_card( $item_id );
			}
			?>
		</div>
		<?php
	}

	/**
	 * Gets the heading text for the extensions page.
	 *
	 * @since 3.1.1
	 * @return string
	 */
	protected function get_heading_text() {
		return ! edd_is_inactive_pro() ? __( 'Pro Available Extensions', 'easy-digital-downloads' ) : parent::get_heading_text();
	}

	/**
	 * If a pass hasn't been saved, show the text offering to add it.
	 *
	 * @since 3.1.1
	 * @return void
	 */
	protected function show_missing_key_question() {
		if ( ! $this->can_show_pass_refresh() ) {
			return;
		}
		printf(
			wp_kses_post(
				/* translators: 1. opening anchor tag; 2. closing anchor tag. */
				__( 'Missing an extension that you think you should be able to see? %1$sRefresh your extensions now%2$s.', 'easy-digital-downloads' )
			),
			'<a href="' . esc_url( add_query_arg( 'edd_action', 'refresh_pass_status' ) ) . '">',
			'</a>'
		);
	}

	/**
	 * Maybe add a break and start showing extensions they cannot install.
	 *
	 * @since 3.1.1
	 * @param array $extension
	 * @return void
	 */
	private function maybe_insert_break( $extension ) {
		if ( ! $this->can_install ) {
			return;
		}
		if ( ! empty( $extension['categories'] ) && ! $this->pass_manager->can_access_categories( $extension['categories'] ) ) {
			$this->can_install = false;
			?>
			</div>
			<div class="edd-extension-manager__unlock">
				<h2><?php esc_html_e( 'Unlock More Extensions', 'easy-digital-downloads' ); ?></h2>
				<p>
					<?php
					$url = edd_link_helper(
						'https://easydigitaldownloads.com/your-account/',
						array(
							'utm_medium'  => 'extensions-page',
							'utm_content' => 'unlock-more',
						)
					);
					printf(
						wp_kses_post(
							/* translators: 1. opening anchor tag; 2. closing anchor tag. */
							__( 'Want to get even more extensions? Upgrade your %1$sEasy Digital Downloads license%2$s and unlock the following extensions.', 'easy-digital-downloads' )
						),
						'<a href="' . esc_url( $url ) . '">',
						'</a>'
					);
					?>
				</p>
			</div>
			<div class="edd-extension-manager__card-group edd-extension-manager__disable">
			<?php
		}
	}

	/**
	 * Update the button parameters.
	 *
	 * @since 3.1.1
	 * @param ProductData $product_data The extension data returned from the Products API.
	 * @param bool|int $item_id
	 * @return array
	 */
	protected function get_button_parameters( ProductData $product_data, $item_id = false ) {
		$button = parent::get_button_parameters( $product_data, $item_id );

		// If the active pass can download the product, show a log in button.
		if ( ! edd_is_inactive_pro() && $this->manager->pass_can_download( $product_data->pass_id ) ) {
			$button['button_text'] = __( 'Log In to Download', 'easy-digital-downloads' );
			unset( $button['button_class'] );
		}

		if ( ! $this->can_install || empty( $product_data->basename ) || $this->manager->is_plugin_installed( $product_data->basename ) || ! $this->manager->pass_can_download( $product_data->pass_id ) || ( is_multisite() && ! is_super_admin() ) ) {
			return $button;
		}

		$button['button_text']  = __( 'Install', 'easy-digital-downloads' );
		$button['action']       = 'install';
		$button['product']      = $item_id;
		$button['button_class'] = 'edd-button__install';
		unset( $button['href'] );
		unset( $button['new_tab'] );

		return $button;
	}

	/**
	 * Checks the current user's capability level.
	 *
	 * @since 3.1.1
	 * @param string $capability
	 * @return bool
	 */
	protected function current_user_can( $capability = 'activate_plugins' ) {
		return $this->can_install && current_user_can( $capability ) && ! edd_is_inactive_pro();
	}

	/**
	 * Gets the upgrade URL for the button.
	 *
	 * @since 3.1.1
	 * @param ProductData $product_data The product data object.
	 * @param int                               $item_id      The item/product ID.
	 * @param bool                              $has_access   Whether the user already has access to the extension (based on pass level).
	 * @return string
	 */
	protected function get_upgrade_url( ProductData $product_data, $item_id, $has_access = false ) {
		if ( $has_access ) {
			$url = 'https://easydigitaldownloads.com/your-account/your-downloads/';
		} else {
			$url = 'https://easydigitaldownloads.com/pricing';
		}

		$utm_parameters = array(
			'utm_medium'  => 'extensions-page',
			'utm_content' => $product_data->slug,
		);

		return edd_link_helper(
			$url,
			$utm_parameters
		);
	}
}
