/* global WPForms, WPFormsActiveCampaign, wpf, wpforms_builder, Choices */
'use strict';

/**
 * WPForms Providers Builder ActiveCampaign module.
 *
 * @since 1.0.0
 */
WPForms.Admin.Builder.Providers.ActiveCampaign = WPForms.Admin.Builder.Providers.ActiveCampaign || ( function( document, window, $ ) {

	/**
	 * Private functions and properties.
	 *
	 * @since 1.0.0
	 *
	 * @type {object}
	 */
	var __private = {

		/**
		 * jQuery selector for holder.
		 *
		 * @since 1.0.0
		 *
		 * @type {object}
		 */
		$holder: null,

		/**
		 * Config contains all configuration properties.
		 *
		 * @since 1.0.0
		 *
		 * @type {object.<string, *>}
		 */
		config: {

			/**
			 * List of ActiveCampaign templates that should be compiled.
			 *
			 * @since 1.0.0
			 * @since 1.2.1 Added `wpforms-activecampaign-builder-content-connection-conditionals` template.
			 *
			 * @type {string[]}
			 */
			templates: [
				'wpforms-activecampaign-builder-content-connection',
				'wpforms-activecampaign-builder-content-connection-subscriber-subscribe',
				'wpforms-activecampaign-builder-content-connection-subscriber-unsubscribe',
				'wpforms-activecampaign-builder-content-connection-subscriber-delete',
				'wpforms-activecampaign-builder-content-connection-event-tracking',
				'wpforms-activecampaign-builder-content-connection-error',
				'wpforms-activecampaign-builder-content-connection-lock',
				'wpforms-activecampaign-builder-content-connection-conditionals',
			],
		},

		/**
		 * Sometimes in DOM we might have placeholders or temporary connection IDs.
		 * We need to replace them with actual values.
		 *
		 * @since 1.0.0
		 *
		 * @param {object} $el jQuery DOM element.
		 * @param {string} connectionId New connection ID to replace to.
		 */
		replaceConnectionIds: function( $el, connectionId ) {

			// Replace old temporary %connection_id% from PHP code with the new one.
			$el.closest( '.wpforms-builder-provider-connection' )
				.find( 'input, textarea, select, label' ).each( function() {

					if ( $( this ).attr( 'name' ) ) {
						$( this ).attr( 'name', $( this ).attr( 'name' ).replace( /%connection_id%/gi, connectionId ) );
					}

					if ( $( this ).attr( 'id' ) ) {
						$( this ).attr( 'id', $( this ).attr( 'id' ).replace( /%connection_id%/gi, connectionId ) );
					}

					if ( $( this ).attr( 'for' ) ) {
						$( this ).attr( 'for', $( this ).attr( 'for' ).replace( /%connection_id%/gi, connectionId ) );
					}

					if ( $( this ).attr( 'data-name' ) ) {
						$( this ).attr( 'data-name', $( this ).attr( 'data-name' ).replace( /%connection_id%/gi, connectionId ) );
					}
				} );
		},

		/**
		 * Sometimes we might need to a get a connection DOM element by its ID.
		 *
		 * @since 1.0.0
		 *
		 * @param {string} connectionId Connection ID to search for a DOM element by.
		 *
		 * @returns {object} jQuery selector for connection.
		 */
		getConnectionById: function( connectionId ) {

			return __private.config.$holder.find( '.wpforms-builder-provider-connection[data-connection_id="' + connectionId + '"]' );
		},

		/**
		 * Whether we have an account ID in a list of all available accounts.
		 *
		 * @since 1.0.0
		 *
		 * @param {string} accountId Connection account ID to check.
		 * @param {Array} accounts Array of objects, usually received from AC API.
		 *
		 * @returns {boolean} True if connection already exists.
		 */
		connectionAccountExists: function( accountId, accounts ) {

			if ( _.isEmpty( accounts ) ) {
				return false;
			}

			// New connections, that have not been saved don't have the account ID yet.
			if ( _.isEmpty( accountId ) ) {
				return true;
			}

			return _.has( accounts, accountId );
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.0.0
	 *
	 * @type {object}
	 */
	var app = {

		/**
		 * Current provider slug.
		 *
		 * @since 1.0.0
		 *
		 * @type {string}
		 */
		provider: 'activecampaign',

		/**
		 * This is a shortcut to the WPForms.Admin.Builder.Providers object,
		 * that handles the parent all-providers functionality.
		 *
		 * @since 1.0.0
		 *
		 * @type {object}
		 */
		Providers: {},

		/**
		 * This is a shortcut to the WPForms.Admin.Builder.Templates object,
		 * that handles all the template management.
		 *
		 * @since 1.0.0
		 *
		 * @type {object}
		 */
		Templates: {},

		/**
		 * This is a shortcut to the WPForms.Admin.Builder.Providers.cache object,
		 * that handles all the cache management.
		 *
		 * @since 1.0.0
		 *
		 * @type {object}
		 */
		Cache: {},

		/**
		 * This is a flag for ready state.
		 *
		 * @since 1.0.0
		 *
		 * @type {boolean}
		 */
		isReady: false,

		/**
		 * Start the engine.
		 *
		 * @since 1.0.0
		 */
		init: function() {

			// Done by reference, so we are not doubling memory usage.
			app.Providers = WPForms.Admin.Builder.Providers;
			app.Templates = WPForms.Admin.Builder.Templates;
			app.Cache     = app.Providers.cache;

			/**
			 * Run initialization on providers panel only.
			 */

			// We are requesting/loading a Providers panel.
			if ( 'providers' === wpf.getQueryString( 'view' ) ) {
				$( '#wpforms-panel-providers' ).on( 'WPForms.Admin.Builder.Providers.ready', app.ready );
			}

			// We have switched to Providers panel.
			$( document ).on( 'wpformsPanelSwitched', function( e, panel ) {
				if ( 'providers' === panel ) {
					app.ready();
				}
			} );
		},

		/**
		 * Initialized once the DOM and Providers are fully loaded.
		 *
		 * @since 1.0.0
		 */
		ready: function() {

			if ( app.isReady ) {
				return;
			}

			// Save a jQuery selector for provider holder.
			__private.config.$holder = $( '#activecampaign-provider' ).find( '.wpforms-builder-provider-body' );

			// Register custom/provider underscorejs templates.
			app.Templates.add( __private.config.templates );

			// Register a handler for Add New Account process.
			app.Providers.ui.account.registerAddHandler( app.provider, app.processAccountAdd );

			// Fire AJAX-request to retrieve the list of all saved connections.
			app.processInitialTemplates();

			// Events registration.
			app.bindUIActions();
			app.bindTriggers();

			// Save a flag for ready state.
			app.isReady = true;
		},

		/**
		 * Compile template with data if any and display them on a page.
		 *
		 * @since 1.0.0
		 */
		processInitialTemplates: function() {

			var tmplError = app.Templates.get( 'wpforms-activecampaign-builder-content-connection-error' ),
				tmplLock  = app.Templates.get( 'wpforms-activecampaign-builder-content-connection-lock' );

			app.connectionsDataLoad( true );

			__private.config.$holder
				.find( '.wpforms-builder-provider-connections' )
				.prepend( tmplError() + tmplLock() );
		},

		/**
		 * Load connections data.
		 *
		 * @since 1.0.0
		 *
		 * @param {boolean} connectionGenerate True if need a connection generation.
		 */
		connectionsDataLoad: function( connectionGenerate ) {

			app.Providers.ajax
				.request( app.provider, {
					data: {
						task: 'connections_get',
					},
				} )
				.done( function( response ) {

					if (
						! response.success ||
						! Object.prototype.hasOwnProperty.call( response.data, 'connections' )
					) {
						return;
					}

					// Save CONNECTIONS to "cache" as a copy.
					app.Cache.set( app.provider, 'connections', jQuery.extend( {}, response.data.connections ) );

					// Save CONDITIONALS to "cache" as a copy.
					app.Cache.set( app.provider, 'conditionals', jQuery.extend( {}, response.data.conditionals ) );

					// Save ACCOUNTS to "cache" as a copy, if we have them.
					if ( ! _.isEmpty( response.data.accounts ) ) {
						app.Cache.set( app.provider, 'accounts', jQuery.extend( {}, response.data.accounts ) );
					}

					__private.config.$holder.trigger( 'connectionsDataLoaded', [ response.data, connectionGenerate ] );
				} );
		},

		/**
		 * Get the template and data for a connection and process it.
		 *
		 * @since 1.0.0
		 *
		 * @param {object} data Connection data.
		 */
		connectionGenerate: function( data ) {

			var tmplConnection  = app.Templates.get( 'wpforms-activecampaign-builder-content-connection' ),
				tmplConditional = $( '#tmpl-wpforms-activecampaign-builder-content-connection-conditionals' ).length ? app.Templates.get( 'wpforms-activecampaign-builder-content-connection-conditionals' ) : app.Templates.get( 'wpforms-providers-builder-content-connection-conditionals' ),
				conditional     = ( Object.prototype.hasOwnProperty.call( data.connection, 'isNew' ) && data.connection.isNew ) ? tmplConditional() : data.conditional,
				accounts        = app.Cache.get( app.provider, 'accounts' ); // Array of account objects.

			/*
			 * We may or may not receive accounts previously.
			 * If yes - render instantly, if no - request them via AJAX.
			 */
			if ( ! _.isEmpty( accounts ) ) {
				if ( __private.connectionAccountExists( data.connection.account_id, accounts ) ) {
					__private.config.$holder
						.find( '.wpforms-builder-provider-connections' )
						.prepend(
							tmplConnection( {
								connection: data.connection,
								accounts: accounts,
								conditional: conditional,
							} )
						);

					// When we are done adding a new connection with its accounts - trigger next steps.
					__private.config.$holder.trigger( 'connectionGenerated', [ data ] );
				}

			} else {

				// We need to get the live list of accounts, as nothing is in cache.
				app.Providers.ajax
					.request( app.provider, {
						data: {
							task: 'accounts_get',
						},
					} )
					.done( function( response ) {

						if (
							! response.success ||
							! Object.prototype.hasOwnProperty.call( response.data, 'accounts' )
						) {
							return;
						}

						// Save ACCOUNTS in "cache" as a copy.
						app.Cache.set( app.provider, 'accounts', response.data.accounts );

						if ( __private.connectionAccountExists( data.connection.account_id, response.data.accounts ) ) {
							__private.config.$holder
								.find( '.wpforms-builder-provider-connections' )
								.prepend(
									tmplConnection( {
										connection: data.connection,
										accounts: response.data.accounts,
										conditional: conditional,
									} )
								);

							// When we are done adding a new connection with its accounts - trigger next steps.
							__private.config.$holder.trigger( 'connectionGenerated', [ data ] );
						}
					} );
			}
		},

		/**
		 * Process the account creation in FormBuilder.
		 *
		 * @since 1.0.0
		 *
		 * @param {object} modal jQuery-Confirm modal object.
		 *
		 * @returns {boolean} False.
		 */
		processAccountAdd: function( modal ) {

			var $apiUrlField = modal.$content.find( 'input[name="api_url"]' ),
				$apiKeyField = modal.$content.find( 'input[name="api_key"]' ),
				$error       = modal.$content.find( '.error' ),
				apiUrl       = $.trim( $apiUrlField.val() ),
				apiKey       = $.trim( $apiKeyField.val() );

			if ( _.isEmpty( apiUrl ) || _.isEmpty( apiKey ) ) {
				$error.show();
				modal.setType( 'red' );

				if ( _.isEmpty( apiUrl ) ) {
					$apiUrlField.addClass( 'wpforms-error' );
				}

				if ( _.isEmpty( apiKey ) ) {
					$apiKeyField.addClass( 'wpforms-error' );
				}

			} else {
				$error.hide();
				modal.setType( 'blue' );
				$apiUrlField.removeClass( 'wpforms-error' );
				$apiKeyField.removeClass( 'wpforms-error' );

				app.Providers.ajax
					.request( app.provider, {
						data: {
							task: 'account_save',
							apiUrl: apiUrl,
							apiKey: apiKey,
							accName: $.trim( modal.$content.find( 'input[name="acc_name"]' ).val() ),
							actid: $.trim( modal.$content.find( 'input[name="actid"]' ).val() ),
							eventKey: $.trim( modal.$content.find( 'input[name="event_key"]' ).val() ),
						},
					} )
					.done( function( response ) {

						if (
							! response.success ||
							(
								Object.prototype.hasOwnProperty.call( response.data, 'error' ) &&
								! _.isEmpty( response.data.error )
							)
						) {
							modal.setType( 'red' );
							$error.html( response.data.error ).show();
							console.log( response );

						} else {

							// Hide `Add New Account` button.
							app.Providers.getProviderHolder( app.provider )
								.find( '.wpforms-builder-provider-title-add' )
								.toggleClass( 'hidden' );
							modal.close();
						}
					} );
			}

			return false;
		},

		/**
		 * Process various events as a response to UI interactions.
		 *
		 * @since 1.0.0
		 */
		bindUIActions: function() {

			var $eHolder = $( '#activecampaign-provider' );

			// CONNECTION: NEW.
			$eHolder.on( 'connectionCreate', function( e, name ) {

				app.connectionCreate( name );
			} );

			// CONNECTION: DELETE.
			$( '.wpforms-builder-provider' ).on( 'connectionDelete', function( e, $curNotification ) {

				if ( ! $curNotification.closest( $eHolder ).length ) {
					return;
				}

				var connectionId = $curNotification.data( 'connection_id' );

				if ( 'undefined' !== typeof connectionId ) {
					var connection = app.Cache.getById( app.provider, 'connections', connectionId ),
						events     = app.Cache.get( app.provider, 'events' );

					if (
						'undefined' !== typeof events &&
						null !== connection &&
						Object.prototype.hasOwnProperty.call( connection, 'account_id' )
					) {
						app.Cache.deleteFrom( app.provider, 'events', connection.account_id );
					}

					app.Cache.deleteFrom( app.provider, 'connections', connectionId );
				}
			} );

			// CONNECTION: ACCOUNT.
			$eHolder.on( 'change', '.js-wpforms-builder-activecampaign-provider-connection-account', app.ui.account.changeCallback );

			// CONNECTION: ACTION.
			$eHolder.on( 'change', '.js-wpforms-builder-activecampaign-provider-connection-action', app.ui.action.changeCallback );

			// CONNECTION: LISTS.
			$eHolder.on( 'change', '.js-wpforms-builder-activecampaign-provider-connection-lists', app.ui.lists.changeCallback );

			// CONNECTION: ADD NEW ITEM.
			$eHolder.on( 'click', '.js-wpforms-builder-activecampaign-provider-add-new-item', app.ui.common.addNewCallback );
		},

		/**
		 * Fire certain events on certain actions, specific for related connections.
		 * These are not directly caused by user manipulations.
		 *
		 * @since 1.0.0
		 */
		bindTriggers: function() {

			__private.config.$holder.on( 'connectionsDataLoaded', function( e, data, connectionGenerate ) {

				connectionGenerate = connectionGenerate || false;

				if ( true === connectionGenerate && ! _.isEmpty( data.accounts ) ) {

					// Does nothing if there are no current connections for a form.
					for ( var connectionId in data.connections ) {

						if ( ! Object.prototype.hasOwnProperty.call( data.connections, connectionId ) ) {
							continue;
						}

						app.connectionGenerate( {
							connection: app.Cache.getById( app.provider, 'connections', connectionId ),
							conditional: app.Cache.getById( app.provider, 'conditionals', connectionId ),
						} );
					}
				}
			} );

			__private.config.$holder.on( 'connectionGenerated', function( e, data ) {

				var $connection = __private.getConnectionById( data.connection.id );

				$( '.js-wpforms-builder-activecampaign-provider-connection-action', $connection ).trigger( 'change', [ $connection ] );
			} );

			__private.config.$holder.on( 'connectionRendered', function( e, data ) {

				var $connection = __private.getConnectionById( data.connection_id );

				$( '.js-wpforms-builder-activecampaign-provider-connection-lists', $connection ).trigger( 'change', [ $connection ] );

				app.loadChoicesJS( $connection );
			} );

			// Properly handle conditional logic DOM modification.
			__private.config.$holder.on( 'DOMNodeInsertedIntoDocument DOMNodeRemovedFromDocument', '.wpforms-conditional-block-panel', function() {

				__private.replaceConnectionIds(
					$( this ),
					$( this ).closest( '.wpforms-builder-provider-connection' ).data( 'connection_id' )
				);
			} );
		},

		/**
		 * Create a connection using the user entered name.
		 *
		 * @since 1.0.0
		 *
		 * @param {string} name Connection name.
		 */
		connectionCreate: function( name ) {

			var connectionId = ( new Date().getTime() ).toString( 16 ),
				connection   = {
					id: connectionId,
					name: name,
					isNew: true,
				};

			app.Cache.addTo( app.provider, 'connections', connectionId, connection );

			app.connectionGenerate( {
				connection: connection,
			} );
		},

		/**
		 * From time to time, based on actions, we need option_id, that is stored in a hidden field.
		 *
		 * @since 1.0.0
		 *
		 * @param {object} $connection jQuery DOM connection element.
		 *
		 * @returns {string} Option ID.
		 */
		connectionGetOptionId: function( $connection ) {

			return $connection.find( '.wpforms-builder-activecampaign-provider-accounts select :selected' ).data( 'option_id' );
		},

		/**
		 * For each connection we should preselect already saved email field.
		 *
		 * @since 1.0.0
		 *
		 * @param {string} connectionId Current connection ID.
		 * @param {object} $connection jQuery DOM connection element.
		 */
		mapEmailsField: function( connectionId, $connection ) {

			var connection = app.Cache.getById( app.provider, 'connections', connectionId );

			// Now we need to map fields from connections to events fields.
			if (
				! _.isEmpty( connection ) &&
				! _.isEmpty( connection.fields ) &&
				(
					! _.isEmpty( connection.fields.email ) ||
					! _.isEmpty( connection.fields.new_email )
				)
			) {
				$( 'select[name="providers[activecampaign][' + connectionId + '][fields][email]"]', $connection ).val(
					connection.fields.email
				);

				$( 'select[name="providers[activecampaign][' + connectionId + '][fields][new_email]"]', $connection ).val(
					connection.fields.new_email
				);
			}
		},

		/**
		 * Load Choices.js library.
		 *
		 * @since 1.2.0
		 *
		 * @param {object} $connection jQuery connection selector.
		 */
		loadChoicesJS: function( $connection ) {

			// Load if function exists.
			if ( typeof window.Choices !== 'function' ) {
				return;
			}

			$connection.find( '.choicesjs-select' ).each( function( idx, el ) {

				var $this = $( el );

				// Return if already initialized.
				if ( 'undefined' !== typeof $this.data( 'choicesjs' ) ) {
					return;
				}

				$this.data( 'choicesjs', new Choices( $this[0], {
					shouldSort: false,
					removeItemButton: true,
				} ) );
			} );
		},

		/**
		 * All methods that modify UI of a page.
		 *
		 * @since 1.0.0
		 */
		ui: {

			/**
			 * Action methods.
			 *
			 * @since 1.0.0
			 */
			action: {

				/**
				 * Callback-function on change event.
				 *
				 * @since 1.0.0
				 *
				 * @param {object} event       Event object.
				 * @param {object} $connection jQuery selector.
				 */
				changeCallback: function( event, $connection ) {

					$connection = $connection || $( this ).closest( '.wpforms-builder-provider-connection' );

					var $el          = $( this ),
						connectionId = $connection.data( 'connection_id' ),
						optionId     = app.connectionGetOptionId( $connection ),
						accountId    = $connection.find( '.wpforms-builder-activecampaign-provider-accounts select' ).val(),
						action       = $el.val();

					$el.removeClass( 'wpforms-error' );
					$( '.wpforms-builder-activecampaign-provider-actions-data', $connection ).empty();
					$( '.wpforms-builder-provider-connection-fields', $connection ).remove();

					app.actions.init( {
						'target': $el,
						'action': action,
						'option_id': optionId,
						'account_id': accountId,
						'connection_id': connectionId,
					} );

					__private.config.$holder.trigger( 'actionChanged', [ connectionId, $connection, action ] );
				},
			},

			/**
			 * Account methods.
			 *
			 * @since 1.0.0
			 */
			account: {

				/**
				 * Callback-function on change event.
				 *
				 * @since 1.0.0
				 */
				changeCallback: function() {

					var $el           = $( this ),
						$connection   = $el.closest( '.wpforms-builder-provider-connection' ),
						$actionSelect = $( '.js-wpforms-builder-activecampaign-provider-connection-action', $connection ),
						account       = $el.val(),
						optionId      = null;

					// Clear all connection data if account was changed.
					$( '.wpforms-builder-activecampaign-provider-actions-data', $connection ).empty();
					$( '.wpforms-builder-provider-connection-fields', $connection ).remove();
					$actionSelect.val( '' );

					// If account is empty.
					if ( '' === account ) {

						// Block Action select box.
						$actionSelect.prop( 'disabled', true );

					} else {
						$el.removeClass( 'wpforms-error' );

						// Unblock Action select box.
						$actionSelect.prop( 'disabled', false );

						// We need to save this, so later we will be able to map easily connection_id to option_id.
						optionId = app.connectionGetOptionId( $connection );
						$connection.find( '.wpforms-builder-provider-connection-option_id' ).val( optionId );
					}

					__private.config.$holder.trigger( 'accountChanged', [ optionId, $connection ] );
				},
			},

			/**
			 * List methods.
			 *
			 * @since 1.0.0
			 */
			lists: {

				/**
				 * Callback-function on change event.
				 *
				 * @since 1.0.0
				 */
				changeCallback: function() {

					var $el          = $( this ),
						listId       = $el.val(),
						$connection  = $el.closest( '.wpforms-builder-provider-connection' ),
						connectionId = $connection.data( 'connection_id' ),
						accountId    = $connection.find( '.wpforms-builder-activecampaign-provider-accounts select' ).val(),
						connection   = app.Cache.getById( app.provider, 'connections', connectionId ),
						customFields,
						cacheKey;

					// Skip if a List wasn't selected.
					if ( null === listId ) {
						return;
					}

					// Remove a Custom Fields html.
					$( '.wpforms-builder-provider-connection-fields', $connection ).remove();

					if (
						Object.prototype.hasOwnProperty.call( connection, 'list' ) &&
						Object.prototype.hasOwnProperty.call( connection.list, 'id' ) &&
						listId !== connection.list.id
					) {
						connection = {
							'id': connectionId,
							'fields_meta': [],
						};
					}

					cacheKey     = connectionId + listId;
					customFields = app.ui.lists.getCustomFields( cacheKey );

					// No cache.
					if ( false === customFields ) {

						app.ui.lists.request( {
							connection: connection,
							accountId: accountId,
							listId: listId,
							cacheKey: cacheKey,
						}, $connection );

						return;
					}

					// Display compiled template with custom fields from cache.
					app.ui.lists.render( {
						connection: connection,
						fields: customFields,
					}, $connection );
				},

				/**
				 * AJAX request for retrieve ActiveCampaign custom fileds.
				 *
				 * @since 1.0.0
				 *
				 * @param {object} args        Arguments.
				 * @param {object} $connection jQuery selector.
				 */
				request: function( args, $connection ) {

					// Make ajax request to get custom fields.
					app.Providers.ajax
						.request( app.provider, {
							data: {
								'task': 'custom_fields_get',
								'connection_account_id': args.accountId,
								'list_id': args.listId,
							},
						} )
						.done( function( response ) {

							if (
								! response.success ||
								! Object.prototype.hasOwnProperty.call( response.data, 'custom_fields' )
							) {
								return;
							}

							// "Register" cache keys.
							if ( 'undefined' === typeof app.Cache.get( app.provider, 'custom_fields' ) ) {
								app.Cache.set( app.provider, 'custom_fields', {} );
							}

							// Save data to "cache" as a copy.
							app.Cache.addTo( app.provider, 'custom_fields', args.cacheKey, response.data.custom_fields );

							// Display compiled template with custom data.
							app.ui.lists.render( {
								connection: args.connection,
								fields: response.data.custom_fields,
							}, $connection );
						} );
				},

				/**
				 * Render HTML with custom fields interface.
				 *
				 * @since 1.0.0
				 *
				 * @param {object} args        Arguments.
				 * @param {object} $connection jQuery selector.
				 */
				render: function( args, $connection ) {
					var tmpl = app.Templates.get( 'wpforms-providers-builder-content-connection-fields' );

					$connection
						.find( '.wpforms-builder-activecampaign-provider-actions-data' )
						.append(
							tmpl( {
								connection: args.connection,
								fields: wpf.getFields(),
								provider: {
									placeholder: WPFormsActiveCampaign.i18n.provider_placeholder,
									slug: app.provider,
									fields: args.fields,
								},
							} )
						);
				},

				/**
				 * Retrieve a cache with custom fields.
				 *
				 * @since 1.0.0
				 *
				 * @param {string} cacheKey Cache key.
				 *
				 * @returns {object|boolean} Cache or false if it does not exist.
				 */
				getCustomFields: function( cacheKey ) {

					var customFields = app.Cache.get( app.provider, 'custom_fields' );

					if ( 'undefined' !== typeof customFields ) {
						customFields = app.Cache.getById( app.provider, 'custom_fields', cacheKey );
					}

					if ( 'undefined' === typeof customFields || null === customFields ) {
						return false;
					}

					return customFields;
				},
			},

			/**
			 * Common methods.
			 *
			 * @since 1.0.0
			 */
			common: {

				/**
				 * Callback-function for Add New item (e.g. tag or event).
				 *
				 * @since 1.0.0
				 */
				addNewCallback: function() {

					var $connection = $( this ).closest( '.wpforms-builder-provider-connection' ),
						$newInput   = $connection.find( '.wpforms-builder-activecampaign-provider-item-input' );

					// Reset currently selected event.
					$( '.js-wpforms-builder-activecampaign-provider-item-select', $connection )
						.prop( 'selectedIndex', 0 )
						.prop( 'disabled', ! $newInput.is( ':visible' ) );

					if ( $newInput.is( ':visible' ) ) {

						// Clear and hide.
						$newInput.hide( 400, function() {
							$newInput.find( 'input' ).val( '' );
						} );

					} else {

						// Display.
						$newInput.show( 'fast', function() {
							$( this ).find( 'input' ).focus();
						} );
					}
				},
			},
		},

		/**
		 * Actions property.
		 *
		 * @since 1.0.0
		 */
		actions: {

			/**
			 * Actions initialization.
			 *
			 * @since 1.0.0
			 *
			 * @param {object} args Arguments.
			 */
			init: function( args ) {

				switch ( args.action ) {

					case 'subscriber_subscribe':
						app.actions.subscriber.subscribe.init( args );
						break;

					case 'subscriber_unsubscribe':
						app.actions.subscriber.unsubscribe.init( args );
						break;

					case 'subscriber_delete':
						app.actions.subscriber.delete.init( args );
						break;

					case 'event_tracking':
						app.actions.tracking.event.init( args );
						break;
				}
			},

			/**
			 * Subscriber property.
			 *
			 * @since 1.0.0
			 */
			subscriber: {

				/**
				 * Subscribe action.
				 *
				 * @since 1.0.0
				 */
				subscribe: {

					/**
					 * Subscribe initialization.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					init: function( args ) {

						var self       = this,
							lists      = app.Cache.get( app.provider, 'lists' ),
							tags       = app.Cache.get( app.provider, 'tags' ),
							tagsDelete = app.Cache.get( app.provider, 'tags_delete' );

						if (
							_.isEmpty( lists ) ||
							_.isEmpty( tags ) ||
							_.isEmpty( tagsDelete ) ||
							! Object.prototype.hasOwnProperty.call( lists, args.account_id ) ||
							! Object.prototype.hasOwnProperty.call( tags, args.account_id ) ||
							! Object.prototype.hasOwnProperty.call( tagsDelete, args.account_id )
						) {

							self.request( args );
						} else {
							self.render( args );
						}
					},

					/**
					 * AJAX request.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					request: function( args ) {

						var self = this;

						// Make ajax request to get lists and tags.
						app.Providers.ajax
							.request( app.provider, {
								data: {
									'task': 'subscribe_data_get',
									'connection_id': args.connection_id,
									'connection_account_id': args.account_id,
									'connection_action': args.action,
									'sources': [ 'lists', 'tags' ],
								},
							} )
							.done( function( response ) {

								if (
									! response.success ||
									_.isEmpty( response.data )
								) {
									return;
								}

								// Cache response data.
								self.cache( response.data, args );

								// Render template.
								self.render( args );
							} );
					},

					/**
					 * Render HTML.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					render: function( args ) {

						// Get template for subscribe.
						var tmpl        = app.Templates.get( 'wpforms-activecampaign-builder-content-connection-subscriber-subscribe' ),
							$connection = args.target.closest( '.wpforms-builder-provider-connection' );

						// Display compiled template with custom data.
						$connection
							.find( '.wpforms-builder-activecampaign-provider-actions-data' )
							.html(
								tmpl( {
									'connection': app.Cache.getById( app.provider, 'connections', args.connection_id ),
									'lists': app.Cache.getById( app.provider, 'lists', args.account_id ),
									'tags': app.Cache.getById( app.provider, 'tags', args.account_id ),
									'tags_delete': app.Cache.getById( app.provider, 'tags_delete', args.account_id ),
								} )
							);

						__private.replaceConnectionIds( args.target, args.connection_id );

						app.mapEmailsField( args.connection_id, $connection );

						__private.config.$holder.trigger( 'connectionRendered', [ args ] );
					},

					/**
					 * Cache response data.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} data Response data.
					 * @param {object} args Arguments.
					 */
					cache: function( data, args ) {

						// "Register" cache keys.
						if ( 'undefined' === typeof app.Cache.get( app.provider, 'lists' ) ) {
							app.Cache.set( app.provider, 'lists', {} );
						}
						if ( 'undefined' === typeof app.Cache.get( app.provider, 'tags' ) ) {
							app.Cache.set( app.provider, 'tags', {} );
						}
						if ( 'undefined' === typeof app.Cache.get( app.provider, 'tags_delete' ) ) {
							app.Cache.set( app.provider, 'tags_delete', {} );
						}

						// Save data to cache by keys.
						if ( Object.prototype.hasOwnProperty.call( data, 'lists' ) ) {
							app.Cache.addTo( app.provider, 'lists', args.account_id, data.lists );
						}
						if ( Object.prototype.hasOwnProperty.call( data, 'tags' ) ) {
							app.Cache.addTo( app.provider, 'tags', args.account_id, data.tags );
						}
						if ( Object.prototype.hasOwnProperty.call( data, 'tags_delete' ) ) {
							app.Cache.addTo( app.provider, 'tags_delete', args.account_id, data.tags_delete );
						}
					},
				},

				/**
				 * Unsubscribe action.
				 *
				 * @since 1.0.0
				 */
				unsubscribe: {

					/**
					 * Unsubscribe initialization.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					init: function( args ) {

						var self  = this,
							lists = app.Cache.get( app.provider, 'lists' );

						if (
							_.isEmpty( lists ) ||
							! Object.prototype.hasOwnProperty.call( lists, args.account_id )
						) {
							self.request( args );

						} else {
							self.render( args );
						}
					},

					/**
					 * AJAX request.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					request: function( args ) {

						var self = this;

						// Make ajax request to get lists.
						app.Providers.ajax
							.request( app.provider, {
								data: {
									'task': 'subscribe_data_get',
									'connection_id': args.connection_id,
									'connection_account_id': args.account_id,
									'connection_action': args.action,
									'sources': [ 'lists' ],
								},
							} )
							.done( function( response ) {

								if (
									! response.success ||
									_.isEmpty( response.data )
								) {
									return;
								}

								// "Register" cache keys.
								if ( 'undefined' === typeof app.Cache.get( app.provider, 'lists' ) ) {
									app.Cache.set( app.provider, 'lists', {} );
								}

								// Save data to cache by keys.
								if ( Object.prototype.hasOwnProperty.call( response.data, 'lists' ) ) {
									app.Cache.addTo( app.provider, 'lists', args.account_id, response.data.lists );
								}

								// Render template.
								self.render( args );
							} );
					},

					/**
					 * Render HTML.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					render: function( args ) {

						// Get template for unsubscribe.
						var tmpl        = app.Templates.get( 'wpforms-activecampaign-builder-content-connection-subscriber-unsubscribe' ),
							$connection = args.target.closest( '.wpforms-builder-provider-connection' );

						// Display compiled template with custom data.
						$connection
							.find( '.wpforms-builder-activecampaign-provider-actions-data' )
							.html(
								tmpl( {
									connection: app.Cache.getById( app.provider, 'connections', args.connection_id ),
									lists: app.Cache.getById( app.provider, 'lists', args.account_id ),
								} )
							);

						__private.replaceConnectionIds( args.target, args.connection_id );

						app.mapEmailsField( args.connection_id, $connection );

						__private.config.$holder.trigger( 'connectionRendered', [ args ] );
					},
				},

				/**
				 * Delete action.
				 *
				 * @since 1.0.0
				 */
				delete: {

					/**
					 * Delete initialization.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					init: function( args ) {

						this.render( args );
					},

					/**
					 * Render HTML.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					render: function( args ) {

						// Get template for delete subscriber.
						var tmpl        = app.Templates.get( 'wpforms-activecampaign-builder-content-connection-subscriber-delete' ),
							$connection = args.target.closest( '.wpforms-builder-provider-connection' );

						// Display compiled template with custom data.
						$connection
							.find( '.wpforms-builder-activecampaign-provider-actions-data' )
							.html( tmpl() );

						__private.replaceConnectionIds( args.target, args.connection_id );

						app.mapEmailsField( args.connection_id, $connection );

						__private.config.$holder.trigger( 'connectionRendered', [ args ] );
					},
				},
			},

			/**
			 * Tracking property.
			 *
			 * @since 1.0.0
			 */
			tracking: {

				/**
				 * Event tracking action.
				 *
				 * @since 1.0.0
				 */
				event: {

					/**
					 * Event tracking initialization.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					init: function( args ) {

						var self   = this,
							events = app.Cache.get( app.provider, 'events' );

						if (
							_.isEmpty( events ) ||
							! Object.prototype.hasOwnProperty.call( events, args.account_id )
						) {
							self.request( args );

						} else {
							self.render( args, {} );
						}
					},

					/**
					 * AJAX request.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 */
					request: function( args ) {

						var self = this;

						// Make ajax request to get events.
						app.Providers.ajax
							.request( app.provider, {
								data: {
									'task': 'events_get',
									'connection_id': args.connection_id,
									'connection_account_id': args.account_id,
								},
							} )
							.done( function( response ) {

								if (
									! response.success ||
									_.isEmpty( response.data )
								) {
									return;
								}

								self.responseSuccess( args, response.data );
							} );
					},

					/**
					 * Function for success response.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 * @param {object} data Response data.
					 */
					responseSuccess: function( args, data ) {

						var self = this;

						if ( // Tracking not configured.
							Object.prototype.hasOwnProperty.call( data, 'configured' ) &&
							true !== data.configured
						) {
							self.modals.setup( args, data );

						} else if ( // Tracking not enabled.
							Object.prototype.hasOwnProperty.call( data, 'enabled' ) &&
							true !== data.enabled
						) {
							self.modals.alert( data );

						} else {

							// Render template.
							self.render( args, data );
						}

					},

					/**
					 * Render HTML.
					 *
					 * @since 1.0.0
					 *
					 * @param {object} args Arguments.
					 * @param {object} data Response data.
					 */
					render: function( args, data ) {

						// Get template for event tracking.
						var tmpl        = app.Templates.get( 'wpforms-activecampaign-builder-content-connection-event-tracking' ),
							$connection = args.target.closest( '.wpforms-builder-provider-connection' );

						// "Register" cache key.
						if ( 'undefined' === typeof app.Cache.get( app.provider, 'events' ) ) {
							app.Cache.set( app.provider, 'events', {} );
						}

						// Save data to cache by key.
						if ( Object.prototype.hasOwnProperty.call( data, 'events' ) ) {
							app.Cache.addTo( app.provider, 'events', args.account_id, data.events );
						}

						// Display compiled template with custom data.
						$connection
							.find( '.wpforms-builder-activecampaign-provider-actions-data' )
							.html( tmpl( {
								connection: app.Cache.getById( app.provider, 'connections', args.connection_id ),
								events: app.Cache.getById( app.provider, 'events', args.account_id ),
							} ) );

						__private.replaceConnectionIds( args.target, args.connection_id );

						app.mapEmailsField( args.connection_id, $connection );

						__private.config.$holder.trigger( 'connectionRendered', [ args ] );
					},

					/**
					 * Modals property.
					 *
					 * @since 1.0.0
					 */
					modals: {

						/**
						 * Show an alert modal.
						 *
						 * @since 1.0.0
						 *
						 * @param {object} data Response data.
						 */
						alert: function( data ) {

							var message = WPFormsActiveCampaign.i18n.event_tracking_disabled;

							if (
								Object.prototype.hasOwnProperty.call( data, 'errors' ) &&
								! _.isEmpty( data.errors.message )
							) {
								message = data.errors.message;
							}

							$.alert( {
								title: wpforms_builder.heads_up,
								content: message,
								icon: 'fa fa-exclamation-circle',
								type: 'orange',
								boxWidth: '425px',
								buttons: {
									confirm: {
										text: wpforms_builder.ok,
										btnClass: 'btn-confirm',
										keys: [ 'enter' ],
									},
								},
							} );
						},

						/**
						 * Show a setup modal.
						 *
						 * @since 1.0.0
						 *
						 * @param {object} args Arguments.
						 * @param {object} data Response data.
						 */
						setup: function( args, data ) {

							var self = this;

							$.confirm( {
								title: data.configured.title,
								content: data.configured.content,
								type: data.configured.type,
								icon: 'fa fa-info-circle',
								buttons: {
									save: {
										text: wpforms_builder.save,
										btnClass: 'btn-confirm',
										keys: [ 'enter' ],
										action: self.configSaveCallback,
										actionArgs: args,
									},
									cancel: {
										text: wpforms_builder.cancel,
										action: self.configCancelCallback,
										actionArgs: args,
									},
								},
								onOpenBefore: function() {

									this.$body.addClass( 'wpforms-providers-account-add-modal' );
								},
							} );
						},

						/**
						 * Callback-function for save Event Tracking ACTID and Key.
						 *
						 * @since 1.0.0
						 *
						 * @param {object} button Button arguments.
						 *
						 * @returns {boolean} False.
						 */
						configSaveCallback: function( button ) {

							var modal          = this,
								args           = button.actionArgs,
								$eventKeyField = modal.$content.find( 'input[name="event_key"]' ),
								$actidField    = modal.$content.find( 'input[name="actid"]' ),
								$error         = modal.$content.find( '.error' ),
								eventKey       = $.trim( $eventKeyField.val() ),
								actid          = $.trim( $actidField.val() );

							if ( _.isEmpty( eventKey ) || _.isEmpty( actid ) ) {

								// Display an error if we have it in DOM.
								$error.show();
								modal.setType( 'red' );

								if ( _.isEmpty( eventKey ) ) {
									$eventKeyField.addClass( 'wpforms-error' );
								}

								if ( _.isEmpty( actid ) ) {
									$actidField.addClass( 'wpforms-error' );
								}

							} else {
								$error.hide();
								modal.setType( 'blue' );
								$eventKeyField.removeClass( 'wpforms-error' );
								$actidField.removeClass( 'wpforms-error' );

								app.Providers.ajax
									.request( app.provider, {
										data: {
											task: 'event_tracking_config_save',
											accountId: args.account_id,
											eventKey: eventKey,
											actid: actid,
										},
									} )
									.done( function( response ) {

										if (
											! response.success ||
											(
												Object.prototype.hasOwnProperty.call( response.data, 'error' ) &&
												! _.isEmpty( response.data.error )
											)
										) {
											modal.setType( 'red' );
											$error.html( response.data.error ).show();
											console.log( response );

										} else {
											modal.close();
											app.actions.tracking.event.request( args );
										}
									} );
							}

							return false;
						},

						/**
						 * Callback-function for cancel Event Tracking ACTID and Key.
						 *
						 * @since 1.0.0
						 *
						 * @param {object} button Button arguments.
						 */
						configCancelCallback: function( button ) {

							var args = button.actionArgs;

							args.target.prop( 'selectedIndex', 0 ).trigger( 'change' );
						},
					},
				},
			},
		},
	};

	// Provide access to public functions/properties.
	return app;

}( document, window, jQuery ) );

// Initialize.
WPForms.Admin.Builder.Providers.ActiveCampaign.init();
