<?php

namespace WPFormsActiveCampaign\Api\V3;

/**
 * Class Client is the main class that communicates with
 * the ActiveCampaign API.
 *
 * @since 1.0.0
 */
class Client {

	/**
	 * Authentication HTTP header name.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const HEADER_AUTH_KEY = 'Api-Token';

	/**
	 * API URL for tracking events.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const EVENT_TRACKING_URL = 'https://trackcmp.net/event';

	/**
	 * URL prefix.
	 *
	 * @since 1.0.0
	 */
	const API_VERSION_URL = '/api/3';

	/**
	 * ActiveCampaign API URL.
	 * Format is https://YOUR_ACCOUNT_NAME.api-us1.com
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $api_url = '';

	/**
	 * ActiveCampaign API token.
	 * Get yours from developer settings.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $api_token = '';

	/**
	 * Event Tracking ACTID.
	 * Get yours from Settings > Tracking > Event Tracking > Event Tracking API.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $event_tracking_actid = '';

	/**
	 * Event Tracking Key.
	 * Get yours from Settings > Tracking > Event Tracking > Event Key.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $event_tracking_key = '';

	/**
	 * Request object for endpoints.
	 *
	 * @since 1.0.0
	 *
	 * @var Http\Request
	 */
	private $client;

	/**
	 * Request object for tracking events.
	 *
	 * @since 1.0.0
	 *
	 * @var Http\Request
	 */
	private $event_tracking_client;

	/**
	 * Client constructor.
	 *
	 * @since 1.0.0
	 *
	 * @param string $api_url              API URL.
	 * @param string $api_token            API token.
	 * @param string $event_tracking_actid Event Tracking ACTID.
	 * @param string $event_tracking_key   Event Tracking Key.
	 */
	public function __construct( $api_url, $api_token, $event_tracking_actid = '', $event_tracking_key = '' ) {

		$this->api_url              = $api_url;
		$this->api_token            = $api_token;
		$this->event_tracking_actid = $event_tracking_actid;
		$this->event_tracking_key   = $event_tracking_key;

		$this->client = new Http\Request(
			$this->get_url(),
			array(
				'headers' => array(
					self::HEADER_AUTH_KEY => $this->api_token,
					'Accept'              => 'application/json',
					'Content-Type'        => 'application/json',
				),
			)
		);

		if (
			! empty( $this->event_tracking_actid ) &&
			! empty( $this->event_tracking_key )
		) {
			$this->event_tracking_client = new Http\Request(
				self::EVENT_TRACKING_URL,
				array(
					'headers' => array(
						'Content-Type' => 'application/x-www-form-urlencoded',
					),
					'body'    => array(
						'actid' => $this->event_tracking_actid,
						'key'   => $this->event_tracking_key,
					),
				)
			);
		}
	}

	/**
	 * Retrieve a client.
	 *
	 * @since 1.0.0
	 *
	 * @return Http\Request
	 */
	public function get_client() {

		return $this->client;
	}

	/**
	 * Retrieve a client for tracking events.
	 *
	 * @since 1.0.0
	 *
	 * @return Http\Request|null
	 */
	public function get_event_tracking_client() {

		return $this->event_tracking_client;
	}

	/**
	 * Retrieve an API URL.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_api_url() {

		return $this->api_url;
	}

	/**
	 * Retrieve an API URL with include a API version
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_url() {

		return rtrim( $this->get_api_url(), '/' ) . self::API_VERSION_URL;
	}

	/**
	 * Retrieve an API token.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_api_token() {

		return $this->api_token;
	}

	/**
	 * Retrieve an Event Tracking ACTID.
	 *
	 * @since 1.0.0
	 *
	 * @return string|null
	 */
	public function get_event_tracking_actid() {

		return $this->event_tracking_actid;
	}

	/**
	 * Retrieve an Event Tracking Key.
	 *
	 * @since 1.0.0
	 *
	 * @return string|null
	 */
	public function get_event_tracking_key() {

		return $this->event_tracking_key;
	}

	/**
	 * Retrieve an instance of Users.
	 *
	 * @since 1.0.0
	 *
	 * @return Endpoints\Users
	 */
	public function users() {

		return new Endpoints\Users( $this );
	}

	/**
	 * Retrieve an instance of Contacts.
	 *
	 * @since 1.0.0
	 *
	 * @return Endpoints\Contacts
	 */
	public function contacts() {

		return new Endpoints\Contacts( $this );
	}

	/**
	 * Retrieve an instance of Lists.
	 *
	 * @since 1.0.0
	 *
	 * @return Endpoints\Lists
	 */
	public function lists() {

		return new Endpoints\Lists( $this );
	}

	/**
	 * Retrieve an instance of Tags.
	 *
	 * @since 1.0.0
	 *
	 * @return Endpoints\Tags
	 */
	public function tags() {

		return new Endpoints\Tags( $this );
	}

	/**
	 * Retrieve an instance of Notes.
	 *
	 * @since 1.0.0
	 *
	 * @return Endpoints\Notes
	 */
	public function notes() {

		return new Endpoints\Notes( $this );
	}

	/**
	 * Retrieve an instance of EventTracking.
	 *
	 * @since 1.0.0
	 *
	 * @return Endpoints\EventTracking
	 */
	public function event_tracking() {

		return new Endpoints\EventTracking( $this );
	}
}
