<?php

namespace WPFormsActiveCampaign\Api\V3\Endpoints;

use WPFormsActiveCampaign\Api\V3\Http\Response;

/**
 * Class Contacts.
 *
 * @see   https://developers.activecampaign.com/reference#contact
 *
 * @since 1.0.0
 */
class Contacts extends Resource {

	/**
	 * Create a contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#create-contact
	 *
	 * @since 1.0.0
	 *
	 * @param array $contact Contact details.
	 *
	 * @return Response|\WP_Error
	 */
	public function create( $contact ) {

		return $this->client
			->get_client()
			->post(
				'/contacts',
				[
					'body' => [
						'contact' => $contact,
					],
				]
			);
	}

	/**
	 * Create or update contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#create-contact-sync
	 *
	 * @since 1.0.0
	 *
	 * @param array $contact Contact details.
	 *
	 * @return Response|\WP_Error
	 */
	public function sync( $contact ) {

		return $this->client
			->get_client()
			->post(
				'/contact/sync',
				[
					'body' => [
						'contact' => $contact,
					],
				]
			);
	}

	/**
	 * Get contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#get-contact
	 *
	 * @since 1.0.0
	 *
	 * @param int $id Contact ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function get( $id ) {

		return $this->client
			->get_client()
			->get( '/contacts/' . $id );
	}

	/**
	 * List all contacts.
	 *
	 * @see   https://developers.activecampaign.com/reference#list-all-contacts
	 *
	 * @since 1.0.0
	 *
	 * @param array $query_params Search/filter params.
	 *
	 * @return Response|\WP_Error
	 */
	public function listAll( $query_params = [] ) {

		$query_params = wp_parse_args( $query_params, [ 'limit' => $this->limit ] );

		return $this->client
			->get_client()
			->get(
				'/contacts',
				[
					'query' => $query_params,
				]
			);
	}

	/**
	 * Update list status for a contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#update-list-status-for-contact
	 *
	 * @since 1.0.0
	 *
	 * @param array $contact_list List of lists of a contact.
	 *
	 * @return Response|\WP_Error
	 */
	public function updateListStatus( $contact_list ) {

		return $this->client
			->get_client()
			->post(
				'/contactLists',
				[
					'body' => [
						'contactList' => $contact_list,
					],
				]
			);
	}

	/**
	 * Update a contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#update-a-contact
	 *
	 * @since 1.0.0
	 *
	 * @param int   $id      Contact ID.
	 * @param array $contact Contact details.
	 *
	 * @return Response|\WP_Error
	 */
	public function update( $id, $contact ) {

		return $this->client
			->get_client()
			->put(
				'/contacts/' . $id,
				[
					'body' => [
						'contact' => $contact,
					],
				]
			);
	}

	/**
	 * Delete a contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#delete-contact
	 *
	 * @since 1.0.0
	 *
	 * @param int $id Contact ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function delete( $id ) {

		return $this->client
			->get_client()
			->delete( '/contacts/' . $id );
	}

	/**
	 * List all automations the contact is in.
	 *
	 * @see   https://developers.activecampaign.com/reference#list-all-contactautomations-for-contact
	 *
	 * @since 1.0.0
	 *
	 * @param int $id Contact ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function listAutomations( $id ) {

		return $this->client
			->get_client()
			->get( '/contacts/' . $id . '/contactAutomations' );
	}

	/**
	 * Get contact tags.
	 *
	 * @since 1.0.0
	 *
	 * @param int $id Contact ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function tags( $id ) {

		return $this->client
			->get_client()
			->get( '/contacts/' . $id . '/contactTags' );
	}

	/**
	 * Add a tag to contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#create-contact-tag
	 *
	 * @since 1.0.0
	 *
	 * @param int $id     Contact ID.
	 * @param int $tag_id Tag ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function tag( $id, $tag_id ) {

		return $this->client
			->get_client()
			->post(
				'/contactTags',
				[
					'body' => [
						'contactTag' => [
							'contact' => $id,
							'tag'     => $tag_id,
						],
					],
				]
			);
	}

	/**
	 * Remove a tag from a contact.
	 *
	 * @see   https://developers.activecampaign.com/reference#delete-contact-tag
	 *
	 * @since 1.0.0
	 *
	 * @param int $contact_tag_id Tag ID for a contact.
	 *
	 * @return Response|\WP_Error
	 */
	public function untag( $contact_tag_id ) {

		return $this->client
			->get_client()
			->delete( '/contactTags/' . $contact_tag_id );
	}

	/**
	 * List all custom fields.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#retrieve-fields-1
	 *
	 * @since 1.0.0
	 *
	 * @param array $query_params List of params.
	 *
	 * @return Response|\WP_Error
	 */
	public function listAllCustomFields( $query_params = [] ) {

		$query_params = wp_parse_args( $query_params, [ 'limit' => $this->limit ] );

		return $this->client
			->get_client()
			->get(
				'/fields',
				[
					'query' => $query_params,
				]
			);
	}

	/**
	 * Create a custom field value.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#create-fieldvalue
	 *
	 * @since 1.0.0
	 *
	 * @param int    $contact_id  Contact ID.
	 * @param int    $field_id    Field ID.
	 * @param string $field_value Field value.
	 *
	 * @return Response|\WP_Error
	 */
	public function createCustomFieldValue( $contact_id, $field_id, $field_value ) {

		return $this->client
			->get_client()
			->post(
				'/fieldValues',
				[
					'body' => [
						'fieldValue' => [
							'contact' => $contact_id,
							'field'   => $field_id,
							'value'   => $field_value,
						],
					],
				]
			);
	}

	/**
	 * Retrieve a custom field value.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#retrieve-a-fieldvalues
	 *
	 * @since 1.0.0
	 *
	 * @param int $field_id Field ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function retrieveCustomFieldValue( $field_id ) {

		return $this->client
			->get_client()
			->get( '/fieldValues/' . $field_id );
	}

	/**
	 * Update a custom field value.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#update-a-custom-field-value-for-contact
	 *
	 * @since 1.0.0
	 *
	 * @param int    $field_value_id ID of the field value.
	 * @param int    $contact_id     Contact ID.
	 * @param int    $field_id       Field ID.
	 * @param string $field_value    Field value.
	 *
	 * @return Response|\WP_Error
	 */
	public function updateCustomFieldValue( $field_value_id, $contact_id, $field_id, $field_value ) {

		return $this->client
			->get_client()
			->put(
				'/fieldValues/' . $field_value_id,
				[
					'body' => [
						'fieldValue' => [
							'contact' => $contact_id,
							'field'   => $field_id,
							'value'   => $field_value,
						],
					],
				]
			);
	}

	/**
	 * Delete a custom field value.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#delete-a-fieldvalue-1
	 *
	 * @since 1.0.0
	 *
	 * @param int $field_value_id ID of the field value.
	 *
	 * @return Response|\WP_Error
	 */
	public function deleteCustomFieldValue( $field_value_id ) {

		return $this->client
			->get_client()
			->delete( '/fieldValues/' . $field_value_id );
	}

	/**
	 * Remove contact from automation.
	 *
	 * @see   https://developers.activecampaign.com/reference#delete-a-contactautomation
	 *
	 * @since 1.0.0
	 *
	 * @param int $contact_automation_id Contact automation ID.
	 *
	 * @return Response|\WP_Error
	 */
	public function removeAutomation( $contact_automation_id ) {

		return $this->client
			->get_client()
			->delete( '/contactAutomation/' . $contact_automation_id );
	}
}
