<?php

namespace WPFormsActiveCampaign\Api\V3\Endpoints;

use WPFormsActiveCampaign\Api\V3\Http\Response;

/**
 * Class EventTracking.
 *
 * @see   https://developers.activecampaign.com/reference#event-tracking
 *
 * @since 1.0.0
 */
class EventTracking extends Resource {

	/**
	 * Retrieve status.
	 *
	 * @see   https://developers.activecampaign.com/reference#retrieve-event-tracking-status
	 *
	 * @since 1.0.0
	 *
	 * @return Response|\WP_Error
	 */
	public function retrieveStatus() {

		return $this->client
			->get_client()
			->get( '/eventTracking' );
	}

	/**
	 * Create a new event.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#create-a-new-event-name-only
	 *
	 * @since 1.0.0
	 *
	 * @param string $event_name Event name.
	 *
	 * @return Response|\WP_Error
	 */
	public function createEvent( $event_name ) {

		return $this->client
			->get_client()
			->post(
				'/eventTrackingEvents',
				[
					'body' => [
						'eventTrackingEvent' => [
							'name' => $event_name,
						],
					],
				]
			);
	}

	/**
	 * Delete an event.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#remove-event-name-only
	 *
	 * @since 1.0.0
	 *
	 * @param string $event_name Event name.
	 *
	 * @return Response|\WP_Error
	 */
	public function deleteEvent( $event_name ) {

		return $this->client
			->get_client()
			->delete( '/eventTrackingEvent/' . $event_name );
	}

	/**
	 * List all events.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#list-all-event-types
	 *
	 * @since 1.0.0
	 *
	 * @param array $query_params Query params.
	 *
	 * @return Response|\WP_Error
	 */
	public function listAllEvents( $query_params = [] ) {

		$query_params = wp_parse_args( $query_params, [ 'limit' => $this->limit ] );

		return $this->client
			->get_client()
			->get(
				'/eventTrackingEvents',
				[
					'query' => $query_params,
				]
			);
	}

	/**
	 * Enable/Disable event tracking.
	 *
	 * @see   https://developers.activecampaign.com/v3/reference#enable-disable-event-tracking
	 *
	 * @since 1.0.0
	 *
	 * @param bool $enabled Whether to enable or disable.
	 *
	 * @return Response|\WP_Error
	 */
	public function toggleEventTracking( $enabled ) {

		return $this->client
			->get_client()
			->put(
				'/eventTracking',
				[
					'body' => [
						'eventTracking' => [
							'enabled' => $enabled,
						],
					],
				]
			);
	}

	/**
	 * Track a specific event.
	 *
	 * @since 1.0.0
	 *
	 * @param string      $event_name Event name.
	 * @param null|array  $event_data Event data.
	 * @param null|string $email      Email address.
	 *
	 * @return Response|\WP_Error
	 */
	public function trackEvent( $event_name, $event_data = null, $email = null ) {

		$form_params = [
			'event' => $event_name,
		];

		if ( null !== $event_data ) {
			$form_params['eventdata'] = $event_data;
		}

		if ( null !== $email ) {
			$form_params['visit'] = wp_json_encode(
				[
					'email' => $email,
				]
			);
		}

		return $this->client
			->get_event_tracking_client()
			->post(
				'',
				[
					'body' => $form_params,
				]
			);
	}
}
