<?php

namespace WPFormsActiveCampaign\Api\V3\Http;

/**
 * Wrapper class for making HTTP requests.
 *
 * @since 1.0.0
 */
class Request {

	/**
	 * User-agent value.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	const LIB_USER_AGENT = 'wpformsactivecampaign-api-v3/1.0.0';

	/**
	 * Base URL.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $base_url = '';

	/**
	 * Parameters.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	protected $parameters = array();

	/**
	 * Request constructor.
	 *
	 * @since 1.0.0
	 *
	 * @param string $base_url Base URL.
	 * @param array  $params   Parameters.
	 */
	public function __construct( $base_url, $params = array() ) {

		$this->base_url   = $base_url;
		$this->parameters = array_merge_recursive( $this->get_default_parameters(), $params );
	}

	/**
	 * Send a GET request.
	 *
	 * @since 1.0.0
	 *
	 * @param string $uri  URI to request.
	 * @param array  $args Arguments for the request.
	 *
	 * @return Response|\WP_Error
	 */
	public function get( $uri, $args = array() ) {

		return $this->request( 'GET', $uri, $args );
	}

	/**
	 * Send a POST request.
	 *
	 * @since 1.0.0
	 *
	 * @param string $uri  URI to request.
	 * @param array  $args Arguments for the request.
	 *
	 * @return Response|\WP_Error
	 */
	public function post( $uri, $args = array() ) {

		return $this->request( 'POST', $uri, $args );
	}

	/**
	 * Send a PUT request.
	 *
	 * @since 1.0.0
	 *
	 * @param string $uri  URI to request.
	 * @param array  $args Arguments for the request.
	 *
	 * @return Response|\WP_Error
	 */
	public function put( $uri, $args = array() ) {

		return $this->request( 'PUT', $uri, $args );
	}

	/**
	 * Send a DELETE request.
	 *
	 * @since 1.0.0
	 *
	 * @param string $uri  URI to request.
	 * @param array  $args Arguments for the request.
	 *
	 * @return Response|\WP_Error
	 */
	public function delete( $uri, $args = array() ) {

		return $this->request( 'DELETE', $uri, $args );
	}

	/**
	 * Send a request based on method (main interface).
	 *
	 * @since 1.0.0
	 *
	 * @param string $method Request method.
	 * @param string $uri    URI to request.
	 * @param array  $args   Options for the request.
	 *
	 * @return Response|\WP_Error
	 */
	public function request( $method, $uri, $args ) {

		// Prepare URL.
		$url = $this->get_base_url() . $uri;

		// Merge options.
		$options = array_merge_recursive( $this->parameters, $args );

		// Set a request method.
		$options['method'] = $method;

		// Prepare body - API expect a JSON format.
		if (
			! empty( $options['headers']['Content-Type'] ) &&
			'application/x-www-form-urlencoded' !== $options['headers']['Content-Type'] &&
			! empty( $options['body'] )
		) {
			$options['body'] = wp_json_encode( $options['body'] );
		}

		// Pass a query data to body.
		if ( ! empty( $options['query'] ) && 'GET' === $method ) {
			$options['body'] = $options['query'];
		}

		/**
		 * Filter a request options before it's sent.
		 *
		 * @since 1.0.0
		 *
		 * @param array   $options  Request options.
		 * @param string  $method   Request method.
		 * @param string  $uri      Request URI.
		 * @param Request $instance Instance of Request class.
		 */
		$options = apply_filters( 'wpformsactivecampaign_api_v3_https_request_request_options', $options, $method, $uri, $this );

		// Retrieve the raw response from a safe HTTP request.
		$response = wp_safe_remote_request( $url, $options );

		return new Response( $response );
	}

	/**
	 * Retrieve a base URL.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_base_url() {

		return $this->base_url;
	}

	/**
	 * Set a parameter by key.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key   Parameter key.
	 * @param mixed  $value Parameter value.
	 */
	public function set_parameter( $key, $value ) {

		$this->parameters[ $key ] = $value;
	}

	/**
	 * Retrieve parameters.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_parameters() {

		return $this->parameters;
	}

	/**
	 * Retrieve default parameters for request.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_default_parameters() {

		/**
		 * Request parameters.
		 *
		 * @since 1.0.0
		 *
		 * @param array   $defaults Request arguments.
		 * @param Request $instance Instance of Request class.
		 */
		return apply_filters(
			'wpformsactivecampaign_api_v3_http_request_get_default_parameters',
			[
				'method'      => 'GET',
				'timeout'     => 30,
				'httpversion' => '1.1',
				'user-agent'  => self::LIB_USER_AGENT,
			],
			$this
		);
	}
}
