<?php

namespace WPFormsActiveCampaign\Api\V3\Http;

/**
 * Wrapper class for parse responses.
 *
 * @since 1.0.0
 */
class Response {

	/**
	 * Input data.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	protected $input = array();

	/**
	 * Output data.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	protected $output = array();

	/**
	 * Request constructor.
	 *
	 * @since 1.0.0
	 *
	 * @param array|\WP_Error $input The response data.
	 */
	public function __construct( $input ) {

		$this->input  = $input;
		$this->output = array(
			'success'     => false,
			'status_code' => '',
			'message'     => '',
			'data'        => '',
		);
	}

	/**
	 * Return a result output array.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get() {

		$is_wp_error = is_wp_error( $this->input );
		$message     = $is_wp_error ? $this->input->get_error_message() : $this->get_response_message();
		$status_code = $is_wp_error ? $this->input->get_error_code() : $this->get_response_code();
		$data        = $is_wp_error ? $this->input->get_error_data() : $this->get_body();

		$this->set_output( 'status_code', $status_code );
		$this->set_output( 'message', $message );
		$this->set_output( 'data', $data );

		if (
			empty( $data ) ||
			$this->has_errors()
		) {
			$this->set_output( 'success', false );

		} else {
			$this->set_output( 'success', true );
		}

		return $this->get_output();
	}

	/**
	 * Retrieve only the response code from the raw response.
	 *
	 * @since 1.0.0
	 *
	 * @return int|string The response code as an integer.
	 */
	public function get_response_code() {

		return wp_remote_retrieve_response_code( $this->input );
	}

	/**
	 * Retrieve only the response message from the raw response.
	 *
	 * @since 1.0.0
	 *
	 * @return string The response message.
	 */
	public function get_response_message() {

		return wp_remote_retrieve_response_message( $this->input );
	}

	/**
	 * Retrieve only the body from the raw response.
	 *
	 * @since 1.0.0
	 *
	 * @param bool $json_decode Whether to decode JSON or not.
	 *
	 * @return string The body of the response.
	 */
	public function get_body( $json_decode = true ) {

		$body = wp_remote_retrieve_body( $this->input );

		if ( $json_decode ) {
			$body = json_decode( $body, true );
		}

		return $body;
	}

	/**
	 * Whether we received errors in the response.
	 *
	 * @since 1.0.0
	 *
	 * @return bool True if response has errors.
	 */
	public function has_errors() {

		return is_wp_error( $this->input ) || ! in_array( $this->get_response_code(), array( 200, 201 ), true );
	}

	/**
	 * Set a value by key in output array.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key   Output key.
	 * @param mixed  $value Output value.
	 */
	public function set_output( $key, $value ) {

		$this->output[ $key ] = $value;
	}

	/**
	 * Retrieve an output array.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_output() {

		return $this->output;
	}

	/**
	 * Retrieve an input array.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function get_input() {

		return $this->input;
	}
}
