<?php

namespace WPFormsActiveCampaign\Provider;

use \WPFormsActiveCampaign\Api\V3 as AC;
use WPForms\Providers\Provider\Settings\FormBuilder as FormBuilderAbstract;

/**
 * Class FormBuilder handles functionality inside the form builder.
 *
 * @since 1.0.0
 */
class FormBuilder extends FormBuilderAbstract {

	/**
	 * Register all hooks (actions and filters).
	 *
	 * @since 1.0.0
	 */
	protected function init_hooks() {

		parent::init_hooks();

		add_filter( 'wpforms_save_form_args', [ $this, 'save_form_args' ], 11, 3 );

		add_filter( 'wpforms_providers_settings_builder_ajax_connections_get_' . $this->core->slug, [ $this, 'ajax_connections_get' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_accounts_get_' . $this->core->slug, [ $this, 'ajax_accounts_get' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_subscribe_data_get_' . $this->core->slug, [ $this, 'ajax_subscribe_data_get' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_custom_fields_get_' . $this->core->slug, [ $this, 'ajax_custom_fields_get' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_events_get_' . $this->core->slug, [ $this, 'ajax_events_get' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_account_template_get_' . $this->core->slug, [ $this, 'ajax_account_template_get' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_account_save_' . $this->core->slug, [ $this, 'ajax_account_save' ] );
		add_filter( 'wpforms_providers_settings_builder_ajax_event_tracking_config_save_' . $this->core->slug, [ $this, 'ajax_event_tracking_config_save' ] );

		add_action( 'wpforms_builder_enqueues', [ $this, 'enqueue_inline_css' ], 10 );
	}

	/**
	 * Preprocess provider data before saving it in form_data when editing form.
	 *
	 * @since 1.0.0
	 *
	 * @param array $form Form array, usable with wp_update_post.
	 * @param array $data Data retrieved from $_POST and processed.
	 * @param array $args Empty by default, may have custom data not intended to be saved, but used for processing.
	 *
	 * @return array
	 */
	public function save_form_args( $form, $data, $args ) {

		// Get a filtered form content.
		$form_data = json_decode( stripslashes( $form['post_content'] ), true );

		// Provider exists.
		if ( ! empty( $form_data['providers'][ $this->core->slug ] ) ) {
			$modified_post_content = $this->modify_form_data( $form_data );

			if ( ! empty( $modified_post_content ) ) {
				$form['post_content'] = wpforms_encode( $modified_post_content );

				return $form;
			}
		}

		/*
		 * This part works when modification is locked or current filter was called on NOT Providers panel.
		 * Then we need to restore provider connections from the previous form content.
		 */

		// Get a "previous" form content (current content are still not saved).
		$prev_form = ! empty( $data['id'] ) ? wpforms()->form->get( $data['id'], [ 'content_only' => true ] ) : [];

		if ( ! empty( $prev_form['providers'][ $this->core->slug ] ) ) {
			$provider = $prev_form['providers'][ $this->core->slug ];

			if ( ! isset( $form_data['providers'] ) ) {
				$form_data = array_merge( $form_data, [ 'providers' => [] ] );
			}

			$form_data['providers'] = array_merge( (array) $form_data['providers'], [ $this->core->slug => $provider ] );
			$form['post_content']   = wpforms_encode( $form_data );
		}

		return $form;
	}

	/**
	 * Prepare modifications for form content, if it's not locked.
	 *
	 * @since 1.0.0
	 *
	 * @param array $form_data Form content.
	 *
	 * @return array|null
	 */
	protected function modify_form_data( $form_data ) {

		if (
			isset( $form_data['providers'][ $this->core->slug ]['__lock__'] ) &&
			1 === absint( $form_data['providers'][ $this->core->slug ]['__lock__'] )
		) {
			return null;
		}

		// Modify content as we need, done by reference.
		foreach ( $form_data['providers'][ $this->core->slug ] as $connection_id => &$connection ) {

			if ( '__lock__' === $connection_id ) {
				unset( $form_data['providers'][ $this->core->slug ]['__lock__'] );
				continue;
			}

			// It will be easier in templates to reuse this connection_id.
			$connection['id'] = $connection_id;

			// Save connection based on action.
			$connection = $this->modify_form_data_connection( $connection );
		}
		unset( $connection );

		return $form_data;
	}

	/**
	 * Make required changes to connection and save it.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array
	 */
	protected function modify_form_data_connection( $connection ) {

		if ( empty( $connection['action'] ) ) {
			return $connection;
		}

		switch ( $connection['action'] ) {

			case 'subscriber_subscribe':
				$connection = $this->modify_subscriber_subscribe_connection( $connection );
				break;

			case 'subscriber_unsubscribe':
				$connection = $this->modify_subscriber_unsubscribe_connection( $connection );
				break;

			case 'event_tracking':
				$connection = $this->modify_event_tracking_connection( $connection );
				break;
		}

		return $connection;
	}

	/**
	 * Modify a connection with `Subscriber: Create or Update` action.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array Modified connection data.
	 */
	protected function modify_subscriber_subscribe_connection( $connection ) {

		if ( ! empty( $connection['list']['id'] ) ) {
			$connection['list']['id'] = sanitize_key( $connection['list']['id'] );
		}

		// Subscribe status: 0 - Unconfirmed; 1 - Active; 2 - Unsubscribed.
		if ( isset( $connection['subscribe_status']['id'] ) ) {
			$connection['subscribe_status']['id'] = absint( $connection['subscribe_status']['id'] );
		}

		// Prepare tags.
		$connection['tags'] = $this->sanitize_tags_post( $connection );

		// Note.
		if ( ! empty( $connection['note'] ) ) {
			$connection['note'] = sanitize_textarea_field( $connection['note'] );
		}

		// Make fields meta field_id an integer, sanitize name.
		$connection['fields_meta'] = $this->sanitize_fields_meta( $connection );

		return $connection;
	}

	/**
	 * Modify a connection with `Subscriber: Unsubscribe` action.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array Modified connection data.
	 */
	protected function modify_subscriber_unsubscribe_connection( $connection ) {

		if ( ! empty( $connection['list']['id'] ) ) {
			$connection['list']['id'] = sanitize_key( $connection['list']['id'] );
		}

		return $connection;
	}

	/**
	 * Modify a connection with `Event Tracking` action.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array Modified connection data.
	 */
	protected function modify_event_tracking_connection( $connection ) {

		// Sanitize current value.
		if ( ! empty( $connection['events']['name'] ) ) {
			$connection['events']['name'] = $this->sanitize_event_name( $connection['events']['name'] );
		}

		// If user provided a new one - re-save it.
		if ( ! empty( $connection['events']['new'] ) ) {
			$connection['events']['name'] = $this->sanitize_event_name( $connection['events']['new'] );
			$connection['events']['new']  = 1;
		}

		return $connection;
	}

	/**
	 * Get the list of all saved connections.
	 *
	 * @since 1.0.0
	 *
	 * @return array Return null on any kind of error. Array of data otherwise.
	 */
	public function ajax_connections_get() {

		$connections = [
			'connections'  => isset( $this->form_data['providers'][ $this->core->slug ] ) ? array_reverse( $this->form_data['providers'][ $this->core->slug ], true ) : [],
			'conditionals' => [],
		];

		// Get conditional logic for each connection_id.
		foreach ( $connections['connections'] as $connection ) {

			if ( empty( $connection['id'] ) ) {
				continue;
			}

			// This will either return an empty placeholder or complete set of rules, as a DOM.
			$connections['conditionals'][ $connection['id'] ] = wpforms_conditional_logic()->builder_block(
				[
					'form'       => $this->form_data,
					'type'       => 'panel',
					'parent'     => 'providers',
					'panel'      => $this->core->slug,
					'subsection' => $connection['id'],
					'reference'  => esc_html__( 'Marketing provider connection', 'wpforms-activecampaign' ),
				],
				false
			);
		}

		// Get accounts as well, so js won't need to make several requests.
		$accounts = $this->ajax_accounts_get();

		return array_merge( $connections, $accounts );
	}

	/**
	 * Get the list of all accounts for all API Tokens that might have been saved.
	 *
	 * @since 1.0.0
	 *
	 * @return array May return an empty sub-array.
	 */
	public function ajax_accounts_get() {

		// Check a cache.
		$cache = get_transient( 'wpforms_providers_' . $this->core->slug . '_ajax_accounts_get' );

		// Retrieve accounts from cache.
		if ( is_array( $cache ) && isset( $cache['accounts'] ) ) {
			return $cache;
		}

		// If no cache - preparing to make real external requests.
		$data     = [ 'accounts' => [] ];
		$accounts = $this->get_accounts_data();

		if ( ! empty( $accounts ) ) {
			$data['accounts'] = $accounts;
		}

		// Save accounts to cache.
		if ( ! empty( $data['accounts'] ) ) {
			set_transient( 'wpforms_providers_' . $this->core->slug . '_ajax_accounts_get', $data, 12 * HOUR_IN_SECONDS );
		}

		return $data;
	}

	/**
	 * Retrieve accounts data, that are registered in addon.
	 *
	 * @since 1.0.0
	 *
	 * @return array Array with accounts (users) data.
	 */
	public function get_accounts_data() {

		$providers = wpforms_get_providers_options();
		$data      = [];

		if ( empty( $providers[ $this->core->slug ] ) ) {
			return $data;
		}

		$update_required = false;

		// We might have several different API tokens.
		foreach ( $providers[ $this->core->slug ] as $option_id => $option ) {

			if (
				empty( $option['api_url'] ) ||
				empty( $option['api_key'] )
			) {
				continue;
			}

			// API call.
			$api_client = $this->get_api_client( $option['api_url'], $option['api_key'] );
			$response   = $api_client->users()->me()->get();

			// Code error 402 - payment required.
			// If API key expired (free trial) this status (without any data) will return for all API calls.
			// That's why we remove it from saved accounts.
			if ( ! empty( $response['status_code'] ) && 402 === (int) $response['status_code'] ) {
				unset( $providers[ $this->core->slug ][ $option_id ] );
				$update_required = true;
				continue;
			}

			if ( empty( $response['success'] ) ) {
				continue;
			}

			if ( ! isset( $response['data']['user'] ) ) {
				continue;
			}

			$data[ $option_id ] = $response['data']['user'];
		}

		if ( $update_required ) {
			update_option( 'wpforms_providers', $providers );
		}

		return $data;
	}

	/**
	 * Retrieve an ActiveCampaign data (lists, tags, etc.), that is needed for subscribing process.
	 *
	 * @since 1.0.0
	 *
	 * @return array|null Return null on any kind of error. Array of data otherwise.
	 */
	public function ajax_subscribe_data_get() {

		// Provider options.
		$options = wpforms_get_providers_options( $this->core->slug );

		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if (
			empty( $_POST['connection_account_id'] ) ||
			empty( $_POST['connection_id'] ) ||
			empty( $_POST['sources'] )
		) {
			return null;
		}

		$connection_id = sanitize_text_field( wp_unslash( $_POST['connection_id'] ) );
		$account_id    = sanitize_text_field( wp_unslash( $_POST['connection_account_id'] ) );
		$sources       = array_map( 'sanitize_key', wp_unslash( $_POST['sources'] ) );
		// phpcs:enable WordPress.Security.NonceVerification.Missing

		if (
			empty( $options[ $account_id ]['api_url'] ) ||
			empty( $options[ $account_id ]['api_key'] )
		) {
			return null;
		}

		// Prepare an API client.
		$api_client = $this->get_api_client( $options[ $account_id ]['api_url'], $options[ $account_id ]['api_key'] );
		$data       = $this->get_sources_data( $api_client, $connection_id, $sources );

		return ! empty( $data ) ? $data : null;
	}

	/**
	 * Retrieve an ActiveCampaign lists and tags.
	 *
	 * @since 1.0.0
	 *
	 * @param AC\Client $api_client    Client class instance.
	 * @param string    $connection_id Connection ID.
	 * @param array     $sources       Array with source slugs, that we expect to retrieve.
	 *
	 * @return array
	 */
	private function get_sources_data( $api_client, $connection_id, $sources ) {

		$data = [];

		// Retrieve lists.
		if ( in_array( 'lists', $sources, true ) ) {
			$lists = $api_client->lists()
			                    ->retrieve()
			                    ->get();

			if ( $lists['success'] ) {
				$data['lists'] = wp_list_pluck( $lists['data']['lists'], 'name', 'id' );
			}
		}

		// Retrieve tags.
		if ( in_array( 'tags', $sources, true ) ) {
			$data['tags'] = [];

			// If we have custom tags - add them.
			// This is useful when no tags have been sent and ActiveCampaign doesn't know about it yet.
			if ( ! empty( $this->form_data['providers'][ $this->core->slug ][ $connection_id ]['tags']['add'] ) ) {
				$data['tags'] = $this->form_data['providers'][ $this->core->slug ][ $connection_id ]['tags']['add'];
			}

			$tags = $api_client->tags()
			                   ->listAll( [ 'type' => 'contact' ] )
			                   ->get();

			if ( $tags['success'] ) {
				$tags                = wp_list_pluck( $tags['data']['tags'], 'tag' );
				$data['tags']        = array_unique( array_merge( $tags, (array) $data['tags'] ) );
				$data['tags_delete'] = $tags;
			}
		}

		return $data;
	}

	/**
	 * Retrieve an ActiveCampaign custom fields, filtered by list ID.
	 *
	 * @since 1.0.0
	 *
	 * @return array|null Return null on any kind of error. Array of data otherwise.
	 */
	public function ajax_custom_fields_get() {

		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if (
			empty( $_POST['connection_account_id'] ) ||
			empty( $_POST['list_id'] )
		) {
			return null;
		}

		$account_id = sanitize_text_field( wp_unslash( $_POST['connection_account_id'] ) );
		$list_id    = absint( $_POST['list_id'] );
		// phpcs:enable WordPress.Security.NonceVerification.Missing

		// Provider options.
		$options = wpforms_get_providers_options( $this->core->slug );

		// Prepare an API client.
		if (
			empty( $options[ $account_id ]['api_url'] ) ||
			empty( $options[ $account_id ]['api_key'] )
		) {
			return null;
		}

		$api_client = $this->get_api_client( $options[ $account_id ]['api_url'], $options[ $account_id ]['api_key'] );

		// API call.
		$r = $api_client->contacts()
			->listAllCustomFields( [ 'listfilter' => $list_id ] )
			->get();

		// Request error.
		if ( ! $r['success'] ) {
			return null;
		}

		// Populate with default ActiveCampaign custom fields.
		$data = [
			'custom_fields' => [
				'fullname' => esc_html__( 'Full Name', 'wpforms-activecampaign' ),
				'phone'    => esc_html__( 'Phone Number', 'wpforms-activecampaign' ),
			],
		];

		$data['custom_fields'] += wp_list_pluck( $r['data']['fields'], 'title', 'id' );

		return $data;
	}

	/**
	 * Get the list of all saved custom events from ActiveCampaign.
	 *
	 * @since 1.0.0
	 *
	 * @return array|null Return null on any kind of error. Array of data otherwise.
	 */
	public function ajax_events_get() {

		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if (
			empty( $_POST['connection_account_id'] ) ||
			empty( $_POST['connection_id'] )
		) {
			return null;
		}

		$account_id    = sanitize_text_field( wp_unslash( $_POST['connection_account_id'] ) );
		$connection_id = sanitize_text_field( wp_unslash( $_POST['connection_id'] ) );
		// phpcs:enable WordPress.Security.NonceVerification.Missing

		// Provider options.
		$options = wpforms_get_providers_options( $this->core->slug );
		$data    = [
			'configured' => false,
			'enabled'    => false,
			'events'     => [],
		];

		// If we can't connect to get events, return early with a notice to a user.
		if (
			empty( $options[ $account_id ]['actid'] ) ||
			empty( $options[ $account_id ]['event_key'] )
		) {
			$data['configured'] = [
				'account_id' => $account_id,
				'title'      => esc_html__( 'Event Tracking Configuration', 'wpforms-activecampaign' ),
				'content'    => $this->get_event_tracking_config_fields( $options[ $account_id ]['label'] ),
				'type'       => 'blue',
			];

			return $data;
		}

		$data['configured'] = true;

		// If we have custom event name - add it.
		// This is useful when no events have been sent and ActiveCampaign doesn't know about it yet.
		if ( ! empty( $this->form_data['providers'][ $this->core->slug ][ $connection_id ]['events']['name'] ) ) {
			$data['events'] = [ $this->form_data['providers'][ $this->core->slug ][ $connection_id ]['events']['name'] ];
		}

		// API client.
		$api_client = $this->get_api_client( $options[ $account_id ]['api_url'], $options[ $account_id ]['api_key'], $options[ $account_id ]['actid'], $options[ $account_id ]['event_key'] );

		// API call - retrieve a Tracking status.
		$status = $api_client->event_tracking()
		                     ->retrieveStatus()
		                     ->get();

		// Event tracking should be enabled in ActiveCampaign account.
		if (
			! $status['success'] ||
			empty( $status['data']['eventTracking']['enabled'] )
		) {
			$data['errors'] = $status;

			return $data;
		}

		$data['enabled'] = wp_validate_boolean( $status['data']['eventTracking']['enabled'] );

		// API call - retrieve registered events.
		$events = $api_client->event_tracking()
		                     ->listAllEvents()
		                     ->get();

		if ( ! $events['success'] ) {
			$data['errors'] = $events;

			return $data;
		}

		$events         = wp_list_pluck( $events['data']['eventTrackingEvents'], 'name' );
		$data['events'] = array_unique( array_merge( (array) $data['events'], (array) $events ) );

		return $data;
	}

	/**
	 * Content for Add New Account modal.
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	public function ajax_account_template_get() {

		return [
			'title'   => esc_html__( 'New ActiveCampaign Account', 'wpforms-activecampaign' ),
			'content' => Core::get_add_new_account_fields_html(),
			'type'    => 'blue',
		];
	}

	/**
	 * Save the data for a new account and validate it.
	 *
	 * @since 1.0.0
	 *
	 * @return array|null Return null on any kind of error. Array of data otherwise.
	 */
	public function ajax_account_save() {

		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if ( empty( $_POST['apiUrl'] ) || empty( $_POST['apiKey'] ) ) {
			return null;
		}

		// Sanitization and slashing.
		$api_url   = esc_url_raw( wp_unslash( $_POST['apiUrl'] ) );
		$api_key   = sanitize_key( wp_unslash( $_POST['apiKey'] ) );
		$acc_name  = ! empty( $_POST['accName'] ) ? sanitize_key( wp_unslash( $_POST['accName'] ) ) : '';
		$actid     = ! empty( $_POST['actid'] ) ? sanitize_key( wp_unslash( $_POST['actid'] ) ) : '';
		$event_key = ! empty( $_POST['eventKey'] ) ? sanitize_key( wp_unslash( $_POST['eventKey'] ) ) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Missing

		// API call - retrieve the logged-in user.
		$user = $this->get_api_client( $api_url, $api_key )
		             ->users()->me()
		             ->get();

		if ( ! $user['success'] ) {
			return [
				/* translators: %s - response message from the API. */
				'error' => sprintf( esc_html__( 'ActiveCampaign API returned an error: %s', 'wpforms-activecampaign' ), esc_html( $user['message'] ) ),
			];
		}

		$user       = $user['data']['user'];
		$option_key = uniqid();

		if ( empty( $acc_name ) ) {
			$acc_name = sanitize_text_field( $user['email'] );
		}

		// Save this account.
		wpforms_update_providers_options(
			$this->core->slug,
			[
				'api_url'   => $api_url,
				'api_key'   => $api_key,
				'actid'     => $actid,
				'event_key' => $event_key,
				'label'     => $acc_name,
				'date'      => time(),
			],
			$option_key
		);

		// Update a cache.
		$cache = get_transient( 'wpforms_providers_' . $this->core->slug . '_ajax_accounts_get' );

		if ( empty( $cache ) ) {
			$cache = [ 'accounts' => [] ];
		}

		$cache['accounts'][ $option_key ] = $user;

		set_transient( 'wpforms_providers_' . $this->core->slug . '_ajax_accounts_get', $cache, 12 * HOUR_IN_SECONDS );

		return [];
	}

	/**
	 * Save the data for an Event Tracking and validate it.
	 *
	 * @since 1.0.0
	 *
	 * @return array|null Return null on any kind of error. Array of data otherwise.
	 */
	public function ajax_event_tracking_config_save() {

		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if (
			empty( $_POST['actid'] ) ||
			empty( $_POST['eventKey'] ) ||
			empty( $_POST['accountId'] )
		) {
			return null;
		}

		// Sanitization and slashing.
		$actid      = sanitize_key( wp_unslash( $_POST['actid'] ) );
		$event_key  = sanitize_key( wp_unslash( $_POST['eventKey'] ) );
		$account_id = sanitize_text_field( wp_unslash( $_POST['accountId'] ) );
		// phpcs:enable WordPress.Security.NonceVerification.Missing

		// Provider options.
		$options = wpforms_get_providers_options( $this->core->slug );

		if ( empty( $options[ $account_id ] ) ) {
			return null;
		}

		$new_option = array_merge(
			$options[ $account_id ],
			[
				'actid'     => $actid,
				'event_key' => $event_key,
			]
		);

		// Save this account.
		wpforms_update_providers_options(
			$this->core->slug,
			$new_option,
			$account_id
		);

		return [];
	}

	/**
	 * Sanitize and retrieve fields meta.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array
	 */
	protected function sanitize_fields_meta( $connection ) {

		$fields_meta = [];

		if ( empty( $connection['fields_meta'] ) ) {
			return $fields_meta;
		}

		foreach ( $connection['fields_meta'] as $id => $property ) {

			if ( ! isset( $property['name'], $property['field_id'] ) ) {
				continue;
			}

			$name = sanitize_key( $property['name'] );

			// Do not allow to save empty property name.
			if ( empty( $name ) ) {
				continue;
			}

			$fields_meta[] = [
				'name'     => $name,
				'field_id' => (int) $property['field_id'],
			];
		}

		return $fields_meta;
	}

	/**
	 * Sanitize and retrieve tags.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array
	 */
	protected function sanitize_tags_post( $connection ) {

		$tags = [
			'add'    => [],
			'delete' => [],
			'new'    => [],
		];

		$connection_id = $connection['id'];
		$form_post     = ! empty( $_POST['data'] ) ? json_decode( stripslashes( $_POST['data'] ), true ) : []; // phpcs:ignore

		// Native WPForms saving not supported a multiple selects.
		// We need to get tags data from $_POST variable.
		foreach ( $form_post as $item ) {

			if ( empty( $item['name'] ) ) {
				continue;
			}

			if ( "providers[{$this->core->slug}][{$connection_id}][tags][add][]" === $item['name'] ) {
				$tags['add'][] = sanitize_text_field( $item['value'] );
			}

			if ( "providers[{$this->core->slug}][{$connection_id}][tags][delete][]" === $item['name'] ) {
				$tags['delete'][] = sanitize_text_field( $item['value'] );
			}
		}

		// If user provided new tags.
		$tags_new = $this->sanitize_tags_new( $connection );
		if ( ! empty( $tags_new ) ) {
			$tags['add'] = array_unique( array_merge( $tags['add'], $tags_new ) );
			$tags['new'] = 1;
		}

		return $tags;
	}

	/**
	 * Sanitize and retrieve tags, that will be created.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array
	 */
	protected function sanitize_tags_new( $connection ) {

		$result = [];

		// If user provided new tags.
		if ( ! empty( $connection['tags']['new'] ) ) {
			$tags_new = explode( ',', $connection['tags']['new'] );
			$result   = array_map( 'sanitize_text_field', $tags_new );
		}

		return $result;
	}

	/**
	 * Sanitize event name - must consist only of alphanumeric characters.
	 *
	 * @since 1.0.0
	 *
	 * @param string $name Event name.
	 *
	 * @return string
	 */
	protected function sanitize_event_name( $name ) {

		return preg_replace( '/[^a-zA-Z0-9 _\-]/', '', $name );
	}

	/**
	 * Prepare a client instance for API requests.
	 *
	 * @since 1.0.0
	 *
	 * @param string $api_url              API URL.
	 * @param string $api_key              API Key.
	 * @param string $event_tracking_actid Event Tracking ACTID.
	 * @param string $event_tracking_key   Event Tracking Key.
	 *
	 * @return AC\Client
	 */
	protected function get_api_client( $api_url, $api_key, $event_tracking_actid = '', $event_tracking_key = '' ) {

		return new AC\Client( $api_url, $api_key, $event_tracking_actid, $event_tracking_key );
	}

	/**
	 * Use this method to register own templates for form builder.
	 * Make sure, that you have `tmpl-` in template name in `<script id="tmpl-*">`.
	 *
	 * @since 1.0.0
	 */
	public function builder_custom_templates() {

		?>

		<!-- Single AC connection. -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection">
			<?php $this->get_underscore_template( 'connection' ); ?>
		</script>

		<!-- Single AC connection block: SUBSCRIBER - CREATE / UPDATE -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection-subscriber-subscribe">
			<?php $this->get_underscore_template( 'subscriber-subscribe' ); ?>
		</script>

		<!-- Single AC connection block: SUBSCRIBER - UNSUBSCRIBE -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection-subscriber-unsubscribe">
			<?php $this->get_underscore_template( 'subscriber-unsubscribe' ); ?>
		</script>

		<!-- Single AC connection block: SUBSCRIBER - DELETE -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection-subscriber-delete">
			<?php $this->get_underscore_template( 'subscriber-delete' ); ?>
		</script>

		<!-- Single AC connection block: EVENT TRACKING -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection-event-tracking">
			<?php $this->get_underscore_template( 'event-tracking' ); ?>
		</script>

		<!-- Single AC connection block: ERROR -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection-error">
			<?php $this->get_underscore_template( 'error' ); ?>
		</script>

		<!-- Single AC connection block: LOCK -->
		<script type="text/html" id="tmpl-wpforms-<?php echo esc_attr( $this->core->slug ); ?>-builder-content-connection-lock">
			<?php $this->get_underscore_template( 'lock' ); ?>
		</script>

		<?php
	}

	/**
	 * Display a generated field with all markup for email selection.
	 * Used internally in templates.
	 *
	 * @since 1.0.0
	 *
	 * @param string $type Is it for a main email field or a secondary.
	 */
	protected function display_email_field( $type = 'main' ) {

		switch ( $type ) {
			case 'main':
				wpforms_panel_field(
					'select',
					'activecampaign',
					'email',
					$this->form_data,
					esc_html__( 'Subscriber Email', 'wpforms-activecampaign' ),
					[
						'parent'        => 'providers',
						'field_name'    => 'providers[activecampaign][%connection_id%][fields][email]',
						'field_map'     => [ 'email' ],
						'placeholder'   => esc_html__( '--- Select Email Field ---', 'wpforms-activecampaign' ),
						'after_tooltip' => '<span class="required">*</span>',
						'input_class'   => 'wpforms-required',
						'input_id'      => 'wpforms-panel-field-' . $this->core->slug . '-%connection_id%-email',
					]
				);
				break;

			case 'new':
				wpforms_panel_field(
					'select',
					'activecampaign',
					'new_email',
					$this->form_data,
					esc_html__( 'New Subscriber Email', 'wpforms-activecampaign' ),
					[
						'parent'      => 'providers',
						'field_name'  => 'providers[activecampaign][%connection_id%][fields][new_email]',
						'field_map'   => [ 'email' ],
						'placeholder' => esc_attr__( '--- Select New Email Field ---', 'wpforms-activecampaign' ),
						'input_id'    => 'wpforms-panel-field-' . $this->core->slug . '-%connection_id%-email-new',
					]
				);
				break;
		}
	}

	/**
	 * Enqueue JavaScript and CSS files.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_assets() {

		parent::enqueue_assets();

		$min = wpforms_get_min_suffix();

		// Make sure that `choicesjs` was enqueued. If not - enqueue it.
		if ( ! wp_script_is( 'choicesjs', 'enqueued' ) ) {
			wp_enqueue_script(
				'choicesjs',
				WPFORMS_PLUGIN_URL . 'assets/lib/choices.min.js',
				[],
				'9.0.1',
				true
			);
		}

		wp_enqueue_script(
			'wpforms-activecampaign-admin-builder',
			WPFORMS_ACTIVECAMPAIGN_URL . "assets/js/activecampaign-builder{$min}.js",
			[ 'wpforms-admin-builder-providers', 'choicesjs' ],
			WPFORMS_ACTIVECAMPAIGN_VERSION,
			true
		);

		wp_localize_script(
			'wpforms-activecampaign-admin-builder',
			'WPFormsActiveCampaign',
			[
				'i18n' => [
					'event_tracking_disabled' => esc_html__( 'Event Tracking must be enabled in your ActiveCampaign account on "Settings > Tracking > Event Tracking" page.', 'wpforms-activecampaign' ),
					'provider_placeholder'    => esc_html__( '--- Select ActiveCampaign Field ---', 'wpforms-activecampaign' ),
				],
			]
		);
	}

	/**
	 * Enqueue an inline CSS styles.
	 *
	 * @since 1.2.0
	 */
	public function enqueue_inline_css() {

		wp_add_inline_style(
			'wpforms-builder',
			// language=CSS PhpStorm.
			'#wpforms-panel-providers .js-wpforms-builder-activecampaign-provider-item-select {
				margin-bottom: 10px; 
			}'
		);
	}

	/**
	 * Get an Underscore JS template.
	 *
	 * @since 1.0.0
	 *
	 * @param string $name Template file name.
	 */
	protected function get_underscore_template( $name ) {

		$name = sanitize_file_name( $name );

		$template_path = WPFORMS_ACTIVECAMPAIGN_PATH . "templates/tmpl/{$name}.php";

		if ( is_readable( $template_path ) ) {
			require_once $template_path;
		}
	}

	/**
	 * Get the form inputs needed to config an Event Tracking feature.
	 *
	 * @since 1.0.0
	 *
	 * @param string $account_name Account name.
	 *
	 * @return string $html HTML fields.
	 */
	protected function get_event_tracking_config_fields( $account_name ) {

		$template_path = WPFORMS_ACTIVECAMPAIGN_PATH . 'templates/settings/event-tracking-config-fields.php';
		$html          = '';

		if ( is_readable( $template_path ) ) {
			ob_start();
			require_once $template_path;
			$html = ob_get_clean();
		}

		return $html;
	}
}
