<?php

namespace WPFormsActiveCampaign\Provider;

use WPFormsActiveCampaign\Api\V3 as AC;
use WPForms\Providers\Provider\Settings\PageIntegrations as PageIntegrationsAbstract;

/**
 * Class PageIntegrations handles functionality inside the Settings > Integrations page.
 *
 * @since 1.0.0
 */
class PageIntegrations extends PageIntegrationsAbstract {

	/**
	 * AJAX to add a provider from the settings integrations tab.
	 *
	 * @since 1.0.0
	 */
	public function ajax_connect() {

		if ( false === parent::ajax_connect() ) {
			wp_send_json_error();
		}

		$defaults = [
			'api_url'   => '',
			'api_key'   => '',
			'actid'     => '',
			'event_key' => '',
		];

		$creds = wp_unslash( wp_parse_args( $_POST['data'], $defaults ) ); // phpcs:ignore

		if (
			empty( $creds['api_url'] ) ||
			empty( $creds['api_key'] )
		) {
			wp_send_json_error(
				[
					'error_msg' => esc_html__( 'Please provide API URL and Key.', 'wpforms-activecampaign' ),
				]
			);
		}

		// Sanitization.
		$creds['api_url']   = esc_url_raw( $creds['api_url'] );
		$creds['api_key']   = sanitize_key( $creds['api_key'] );
		$creds['actid']     = sanitize_key( $creds['actid'] );
		$creds['event_key'] = sanitize_key( $creds['event_key'] );

		if ( $this->is_already_exist( $creds ) ) {
			wp_send_json_error(
				[
					'error_msg' => esc_html__( 'Account with those credentials has already been added.', 'wpforms-activecampaign' ),
				]
			);
		}

		wp_send_json( $this->api_get_user( $creds ) );
	}

	/**
	 * Check if account with those credentials already exists.
	 *
	 * @since 1.0.0
	 *
	 * @param array $creds Array with credentials for check.
	 *
	 * @return bool True if account already exists, false on error or if doesn't exist.
	 */
	protected function is_already_exist( $creds ) {

		$options = wpforms_get_providers_options( $this->core->slug );

		if ( empty( $options ) ) {
			return false;
		}

		$keys = wp_list_pluck( $options, 'api_key' );
		$urls = wp_list_pluck( $options, 'api_url' );

		$key_exist = array_search( $creds['api_key'], $keys, true );
		$url_exist = array_search( $creds['api_url'], $urls, true );

		return false !== $url_exist && false !== $key_exist;
	}

	/**
	 * Retrieve a response data about the user with provided credentials.
	 *
	 * @since 1.0.0
	 *
	 * @param array $creds Array with user credentials.
	 *
	 * @return array
	 */
	protected function api_get_user( $creds ) {

		$return = [
			'success' => false,
			'data'    => [],
		];

		// API call.
		$api_client = new AC\Client( $creds['api_url'], $creds['api_key'] );
		$user       = $api_client->users()->me()->get();

		// Request error.
		if ( ! $user['success'] ) {
			$return['data']['error_msg'] = esc_html( $user['message'] );

			return $return;
		}

		// Response data.
		if ( empty( $user['data']['user'] ) ) {
			$return['data']['error_msg'] = esc_html__( 'ActiveCampaign API error: URL or token are not valid.', 'wpforms-activecampaign' );

			return $return;
		}

		// Success.
		$return['success']      = true;
		$return['data']['html'] = $this->prepare_result_html_list( $creds, $user['data']['user'] );

		return $return;
	}

	/**
	 * Prepare a HTML for a new connection.
	 *
	 * @since 1.0.0
	 *
	 * @param array $creds Array with user credentials.
	 * @param array $user  User data.
	 *
	 * @return string
	 */
	protected function prepare_result_html_list( $creds, $user ) {

		$label = $creds['acc_name'];

		if ( empty( $label ) ) {
			$label = sanitize_email( $user['email'] );
		}

		$key = uniqid( '', true );

		// Save this connection for the provider.
		wpforms_update_providers_options(
			$this->core->slug,
			[
				'api_url'   => $creds['api_url'],
				'api_key'   => $creds['api_key'],
				'actid'     => $creds['actid'],
				'event_key' => $creds['event_key'],
				'label'     => $label,
				'date'      => time(),
			],
			$key
		);

		$template_path = WPFORMS_ACTIVECAMPAIGN_PATH . 'templates/settings/connected-account.php';
		$list          = '';

		if ( is_readable( $template_path ) ) {
			ob_start();
			require_once $template_path;
			$list = ob_get_clean();
		}

		return $list;
	}

	/**
	 * Display fields that will store ActiveCampaign account details.
	 *
	 * @since 1.0.0
	 */
	protected function display_add_new_connection_fields() {

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo Core::get_add_new_account_fields_html();
	}
}
