<?php

namespace WPFormsActiveCampaign\Provider;

use WPForms\Tasks\Meta;
use WPFormsActiveCampaign\Api\V3\Client as AC_Client;
use WPFormsActiveCampaign\Api\V3\Http\Response as AC_Response;
use WPFormsActiveCampaign\Api\V3\Endpoints\Contacts as AC_Contacts;

/**
 * Class Process handles entries processing using the provider settings and configuration.
 *
 * @since 1.0.0
 */
class Process extends \WPForms\Providers\Provider\Process {

	/**
	 * Async task action: subscribe.
	 *
	 * @since 1.0.0
	 */
	const ACTION_SUBSCRIBE = 'wpforms_activecampaign_process_action_subscribe';
	/**
	 * Async task action: unsubscribe.
	 *
	 * @since 1.0.0
	 */
	const ACTION_UNSUBSCRIBE = 'wpforms_activecampaign_process_action_unsubscribe';
	/**
	 * Async task action: unsubscribe.
	 *
	 * @since 1.0.0
	 */
	const ACTION_DELETE = 'wpforms_activecampaign_process_action_delete';
	/**
	 * Async task action: events.
	 *
	 * @since 1.0.0
	 */
	const ACTION_EVENTS = 'wpforms_activecampaign_process_action_events';

	/**
	 * Connection data.
	 *
	 * @since 1.0.0
	 *
	 * @var array
	 */
	private $connection;

	/**
	 * Main class that communicates with the ActiveCampaign API.
	 *
	 * @since 1.0.0
	 *
	 * @var AC_Client
	 */
	private $api_client;

	/**
	 * Process constructor.
	 *
	 * @since 1.0.0
	 *
	 * @param Core $core Core instance of the provider class.
	 */
	public function __construct( Core $core ) {

		parent::__construct( $core );

		$this->hook();
	}

	/**
	 * Register hooks.
	 *
	 * @since 1.0.0
	 */
	public function hook() {

		// Register async tasks handlers.
		add_action( self::ACTION_SUBSCRIBE, [ $this, 'task_async_action_trigger' ] );
		add_action( self::ACTION_UNSUBSCRIBE, [ $this, 'task_async_action_trigger' ] );
		add_action( self::ACTION_DELETE, [ $this, 'task_async_action_trigger' ] );
		add_action( self::ACTION_EVENTS, [ $this, 'task_async_action_trigger' ] );
	}

	/**
	 * Receive all wpforms_process_complete params and do the actual processing.
	 *
	 * @since 1.0.0
	 *
	 * @param array $fields    Array of form fields.
	 * @param array $entry     Submitted form content.
	 * @param array $form_data Form data and settings.
	 * @param int   $entry_id  ID of a saved entry.
	 */
	public function process( $fields, $entry, $form_data, $entry_id ) {

		if ( empty( $form_data['providers'][ $this->core->slug ] ) ) {
			return;
		}

		$this->fields    = $fields;
		$this->entry     = $entry;
		$this->form_data = $form_data;
		$this->entry_id  = $entry_id;

		$this->process_each_connection();
	}

	/**
	 * Iteration loop for connections - call action for each connection.
	 *
	 * @since 1.0.0
	 */
	protected function process_each_connection() {

		foreach ( $this->form_data['providers'][ $this->core->slug ] as $connection_id => $connection ) :
			if ( empty( $connection['fields'] ) ) {
				continue;
			}

			// Make sure that we have an email.
			$email = '';
			if ( isset( $this->fields[ $connection['fields']['email'] ]['value'] ) ) {
				$email = strtolower( $this->fields[ $connection['fields']['email'] ]['value'] );
			}

			if ( empty( $email ) ) {
				continue;
			}

			// Check for conditional logic.
			if ( ! $this->is_conditionals_passed( $connection ) ) {
				continue;
			}

			// Fire a connection action.
			switch ( $connection['action'] ) {

				case 'subscriber_subscribe':
					wpforms()->get( 'tasks' )
					         ->create( self::ACTION_SUBSCRIBE )->async()
					         ->params( $connection, $this->fields, $this->form_data, $this->entry_id )
					         ->register();
					break;

				case 'subscriber_unsubscribe':
					wpforms()->get( 'tasks' )
					         ->create( self::ACTION_UNSUBSCRIBE )->async()
					         ->params( $connection, $this->fields, $this->form_data, $this->entry_id )
					         ->register();
					break;

				case 'subscriber_delete':
					wpforms()->get( 'tasks' )
					         ->create( self::ACTION_DELETE )->async()
					         ->params( $connection, $this->fields, $this->form_data, $this->entry_id )
					         ->register();
					break;

				case 'event_tracking':
					wpforms()->get( 'tasks' )
					         ->create( self::ACTION_EVENTS )->async()
					         ->params( $connection, $this->fields, $this->form_data, $this->entry_id )
					         ->register();
					break;
			}

		endforeach;
	}

	/**
	 * Process Conditional Logic for the provided connection.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return bool False if CL rules stopped the connection execution, true otherwise.
	 */
	protected function is_conditionals_passed( $connection ) {

		$pass = $this->process_conditionals( $this->fields, $this->form_data, $connection );

		// Check for conditional logic.
		if ( ! $pass ) {
			wpforms_log(
				esc_html__( 'Form to ActiveCampaign processing stopped by conditional logic.', 'wpforms-activecampaign' ),
				$this->fields,
				array(
					'type'    => array( 'provider', 'conditional_logic' ),
					'parent'  => $this->entry_id,
					'form_id' => $this->form_data['id'],
				)
			);
		}

		return $pass;
	}

	/**
	 * Process the addon async tasks.
	 *
	 * @since 1.0.0
	 *
	 * @param int $meta_id Task meta ID.
	 */
	public function task_async_action_trigger( $meta_id ) {

		$meta = $this->get_task_meta( $meta_id );

		// We expect a certain type and number of params.
		if ( ! is_array( $meta ) || count( $meta ) !== 4 ) {
			return;
		}

		// We expect a certain meta data structure for this task.
		list( $this->connection, $this->fields, $this->form_data, $this->entry_id ) = $meta;

		$this->api_client = $this->get_api_client();

		if ( $this->api_client === null ) {
			return;
		}

		// Finally, fire the actual action processing.
		switch ( $this->connection['action'] ) {

			case 'subscriber_subscribe':
				$this->task_async_action_subscribe();
				break;

			case 'subscriber_unsubscribe':
				$this->task_async_action_unsubscribe();
				break;

			case 'subscriber_delete':
				$this->task_async_action_delete();
				break;

			case 'event_tracking':
				$this->task_async_action_events();
				break;
		}
	}

	/**
	 * Subscriber: Create or Update action.
	 *
	 * @since 1.0.0
	 */
	protected function task_async_action_subscribe() {

		$new_email  = '';
		$is_updated = false;

		// Make sure that we have a new email.
		if ( isset(
			$this->connection['fields']['new_email'],
			$this->fields[ $this->connection['fields']['new_email'] ]['value']
		) ) {
			$new_email = strtolower( $this->fields[ $this->connection['fields']['new_email'] ]['value'] );
		}

		// Prepare endpoint instance.
		$contact_endpoint = $this->api_client->contacts();

		// Run a contact sync.
		$contact = $this->api_contact_sync( $contact_endpoint, $this->connection );

		if ( empty( $contact ) ) {
			return;
		}

		// We need a contact ID for follow actions.
		$contact_id = absint( $contact['id'] );

		// Check if contact was updated.
		// If it's a new contact - `cdate` and `udate` values are equal.
		if (
			! empty( $contact['udate'] ) &&
			$contact['udate'] !== $contact['cdate']
		) {
			$is_updated = true;
		}

		// Update a contact email - an additional request is required.
		if (
			$is_updated &&
			! empty( $new_email ) &&
			$contact['email'] !== $new_email
		) {
			$args = array(
				'email' => $new_email,
			);

			$response = $contact_endpoint->update( $contact_id, $args );
			$response->get();
			$this->maybe_log_errors( $response, $this->connection );
		}

		// Sometime we are forced to apply delay before next API call.
		// It's really an ActiveCampaign API specialty.
		$this->apply_sleep();

		// Next API calls.
		$this->api_contact_subscribe( $this->api_client, $this->connection, $contact_id, $is_updated );
		$this->api_contact_custom_fields( $this->api_client, $this->connection, $contact_id, $is_updated );
		$this->api_contact_tags_sync( $this->api_client, $this->connection, $contact_id, $is_updated );
		$this->api_contact_note( $this->api_client, $this->connection, $contact_id, $is_updated );
	}

	/**
	 * Subscriber: Unsubscribe action.
	 *
	 * @since 1.0.0
	 */
	protected function task_async_action_unsubscribe() {

		// Skip if no lists passed.
		if ( empty( $this->connection['list']['id'] ) ) {
			return;
		}

		// Prepare endpoint instance.
		$contact_endpoint = $this->api_client->contacts();

		// Search a contact.
		$contact = $this->api_contact_search( $contact_endpoint, $this->connection );

		// Contact not found.
		if ( empty( $contact ) ) {
			return;
		}

		// Additional request for retrieve lists where contact is exists.
		$response     = $contact_endpoint->get( absint( $contact['id'] ) );
		$contact_data = $response->get();

		// Request error.
		if ( $this->maybe_log_errors( $response, $this->connection ) ) {
			return;
		}

		$this->api_contact_unsubscribe( $this->api_client, $this->connection, $contact_data['data'] );
	}

	/**
	 * Subscriber: Delete action.
	 *
	 * @since 1.0.0
	 */
	protected function task_async_action_delete() {

		// Prepare endpoint instance.
		$contact_endpoint = $this->api_client->contacts();

		// Search a contact.
		$contact = $this->api_contact_search( $contact_endpoint, $this->connection );

		// Contact not found.
		if ( empty( $contact ) ) {
			return;
		}

		// API call - delete an existing contact.
		$response = $contact_endpoint->delete( absint( $contact['id'] ) );
		$deleted  = $response->get();

		if ( ! $deleted['success'] ) {
			$this->log_errors( $response->get_input(), $this->connection, $deleted['message'] );
		}
	}

	/**
	 * Event Tracking action processor.
	 *
	 * @since 1.0.0
	 */
	protected function task_async_action_events() {

		$contact = $this->api_contact_search( $this->api_client->contacts(), $this->connection );

		// Create a contact if it doesn't exist.
		if ( empty( $contact ) ) {
			$contact = $this->api_contact_sync( $this->api_client->contacts(), $this->connection );
		}

		$actid     = $this->api_client->get_event_tracking_actid();
		$event_key = $this->api_client->get_event_tracking_key();

		// Make sure, that all require data exists.
		if (
			empty( $contact ) ||
			empty( $actid ) ||
			empty( $event_key ) ||
			empty( $this->connection['events']['name'] )
		) {

			return;
		}

		$event_name = $this->connection['events']['name'];

		// Check a Tracking status.
		if ( ! $this->api_is_event_tracking_enabled( $this->api_client, $this->connection ) ) {

			return;
		}

		// If needed - create a new event.
		if ( ! empty( $this->connection['events']['new'] ) ) {

			// API call - create a new event.
			$r = $this->api_client->event_tracking()->createEvent( $event_name );
			$r->get();
			$this->maybe_log_errors( $r, $this->connection );
		}

		$email = strtolower( $this->fields[ $this->connection['fields']['email'] ]['value'] );

		// API call - track an event.
		$r      = $this->api_client->event_tracking()->trackEvent( $event_name, $this->get_event_data(), $email );
		$result = $r->get();

		if ( ! $result['success'] || ! $result['data']['success'] ) {
			$this->log_errors( $r->get_input(), $this->connection, $result['message'] );
		}
	}

	/**
	 * Get event data.
	 *
	 * @since 1.3.0
	 *
	 * @return string
	 */
	private function get_event_data() {

		$event_data = '';
		$data       = [
			'form_id'         => (int) $this->form_data['id'],
			'connection_name' => esc_html( $this->connection['name'] ),
		];

		if ( ! empty( $this->entry_id ) ) {
			$data = wpforms_array_insert(
				$data,
				[ 'entry_id' => (int) $this->entry_id ],
				'form_id'
			);
		}

		foreach ( $data as $name => $value ) {
			$event_data .= sprintf( '%s: %s,', $name, $value );
		}

		$event_data = rtrim( $event_data, ',' );

		/**
		 * Allow theme and plugin authors to change an event data in tracking process.
		 *
		 * @since 1.0.0.
		 *
		 * @param string $event_data Event data.
		 * @param array  $form_data  Form data.
		 * @param int    $entry_id   Entry ID.
		 * @param array  $connection Connection data.
		 */
		return (string) apply_filters(
			'wpforms_activecampaign_process_event_tracking_data',
			$event_data,
			$this->form_data,
			$this->entry_id,
			$this->connection
		);
	}

	/**
	 * Get task meta data.
	 *
	 * @since 1.0.0
	 *
	 * @param int $meta_id Task meta ID.
	 *
	 * @return array|null Null when no data available.
	 */
	protected function get_task_meta( $meta_id ) {

		$task_meta = new Meta();
		$meta      = $task_meta->get( (int) $meta_id );

		// We should actually receive something.
		if ( empty( $meta ) || empty( $meta->data ) ) {
			return null;
		}

		return $meta->data;
	}

	/**
	 * Below are API related methods and their helpers.
	 */

	/**
	 * Get the API client based on connection and provider options.
	 *
	 * @since 1.0.0
	 *
	 * @return AC_Client|null Null on error.
	 */
	protected function get_api_client() {

		if ( empty( $this->connection['account_id'] ) ) {
			return null;
		}

		$providers_options = $this->get_options();

		// Validate existence of required data - API URL and key.
		if (
			empty( $providers_options[ $this->connection['account_id'] ]['api_url'] ) ||
			empty( $providers_options[ $this->connection['account_id'] ]['api_key'] ) ||
			(
				// For "Event Tracking" additional check: actid and event key.
				( 'event_tracking' === $this->connection['action'] ) &&
				(
					empty( $providers_options[ $this->connection['account_id'] ]['actid'] ) ||
					empty( $providers_options[ $this->connection['account_id'] ]['event_key'] )
				)
			)
		) {

			return null;
		}

		// Prepare an API client.
		return new AC_Client(
			$providers_options[ $this->connection['account_id'] ]['api_url'],
			$providers_options[ $this->connection['account_id'] ]['api_key'],
			$providers_options[ $this->connection['account_id'] ]['actid'],
			$providers_options[ $this->connection['account_id'] ]['event_key']
		);
	}

	/**
	 * Check event tracking status.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client Client class instance.
	 * @param array     $connection Connection data.
	 *
	 * @return bool
	 */
	protected function api_is_event_tracking_enabled( $api_client, $connection ) {

		// API call - retrieve a Tracking status.
		$r           = $api_client->event_tracking()->retrieveStatus();
		$status_data = $r->get();

		if ( $this->maybe_log_errors( $r, $connection ) ) {
			return false;
		}

		// Validate a tracking status.
		$status = wp_validate_boolean( $status_data['data']['eventTracking']['enabled'] );

		// If tracking is NOT enabled.
		if ( ! $status ) {
			$this->log_errors( $r->get_input(), $connection, $status_data['message'] );

			return false;
		}

		return $status;
	}

	/**
	 * Search a contact by email.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Contacts $api_endpoint API endpoint instance.
	 * @param array       $connection   Connection data.
	 *
	 * @return array Contact data.
	 */
	protected function api_contact_search( $api_endpoint, $connection ) {

		// API call - search a contact.
		$args = array(
			'email' => strtolower( $this->fields[ $connection['fields']['email'] ]['value'] ),
		);

		$response = $api_endpoint->listAll( $args );
		$contacts = $response->get();

		// Request error.
		if ( $this->maybe_log_errors( $response, $connection ) ) {
			return [];
		}

		if ( empty( $contacts['data']['contacts'] ) ) {
			return [];
		}

		// Return the first item from search results.
		return reset( $contacts['data']['contacts'] );
	}

	/**
	 * Contact sync: create a new contact or update already existing contact data, but not an email.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Contacts $api_endpoint API endpoint instance.
	 * @param array       $connection   Connection data.
	 *
	 * @return array Contact data. Empty array on error.
	 */
	protected function api_contact_sync( $api_endpoint, $connection ) {

		// Request arguments.
		$args = $this->prepare_contact_sync_args( $connection );

		/**
		 * Allow theme and plugin authors to change an arguments for contact sync request (e.g. change a mapping for default fields).
		 *
		 * @since 1.0.0.
		 *
		 * @param array $args       Request arguments.
		 * @param array $connection Connection data.
		 * @param array $fields     Array of form fields.
		 * @param array $entry      Submitted form content.
		 */
		$args = apply_filters( 'wpforms_activecampaign_api_contact_sync_args', $args, $connection, $this->fields, $this->entry );

		// Create a new contact or update already existing contact data, but not an email.
		$response = $api_endpoint->sync( $args );
		$contact  = $response->get();

		// Request error.
		if ( $this->maybe_log_errors( $response, $connection ) ) {
			return [];
		}

		return $contact['data']['contact'];
	}

	/**
	 * Prepare arguments for contact sync request.
	 *
	 * @since 1.0.0
	 *
	 * @param array $connection Connection data.
	 *
	 * @return array
	 */
	protected function prepare_contact_sync_args( $connection ) {

		$args = array(
			'email' => strtolower( $this->fields[ $connection['fields']['email'] ]['value'] ),
		);

		if ( empty( $connection['fields_meta'] ) ) {
			return $args;
		}

		foreach ( $connection['fields_meta'] as $field_meta ) {

			if ( 'phone' === $field_meta['name'] ) {
				$args['phone'] = $this->fields[ $field_meta['field_id'] ]['value'];
				continue;
			}

			if ( 'fullname' === $field_meta['name'] ) {

				if (
					! empty( $this->fields[ $field_meta['field_id'] ]['type'] ) &&
					'name' === $this->fields[ $field_meta['field_id'] ]['type']
				) {
					$args['firstName'] = $this->fields[ $field_meta['field_id'] ]['first'];
					$args['lastName']  = $this->fields[ $field_meta['field_id'] ]['last'];
				}

				// Fallback if no values for `firstName` and `lastName`.
				if ( empty( $args['firstName'] ) && empty( $args['lastName'] ) ) {
					$args['firstName'] = $this->fields[ $field_meta['field_id'] ]['value'];
				}

				continue;
			}
		}

		return $args;
	}

	/**
	 * Add a contact to a list.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $connection Connection data.
	 * @param int       $contact_id Contact ID.
	 * @param bool      $is_updated True if a contact was updated, false - if was created new.
	 */
	protected function api_contact_subscribe( $api_client, $connection, $contact_id, $is_updated ) {

		if ( empty( $connection['list']['id'] ) ) {
			return;
		}

		$contact_data = array();
		if ( $is_updated ) {

			// API call - retrieve full data about contact.
			$r             = $api_client->contacts()->get( $contact_id );
			$_contact_data = $r->get();

			if ( ! $this->maybe_log_errors( $r, $connection ) ) {
				$contact_data = $_contact_data['data'];
			}
		}

		$subscribe_list_id = absint( $connection['list']['id'] );
		$subscribe_status  = isset( $connection['subscribe_status']['id'] ) ? absint( $connection['subscribe_status']['id'] ) : 0;
		$contact_lists     = isset( $contact_data['contactLists'] ) ? array_map( 'absint', wp_list_pluck( $contact_data['contactLists'], 'status', 'list' ) ) : array();

		if ( isset( $contact_lists[ $subscribe_list_id ] ) ) {
			$contact_list_status = $contact_lists[ $subscribe_list_id ];

			// Skip if contact was already subscribed to the list.
			// 0 - Unconfirmed; 1 - Active.
			if ( in_array( $contact_list_status, array( 0, 1 ), true ) ) {
				return;
			}
		}

		// API call - subscribe a contact to a list.
		$args = array(
			'list'    => $subscribe_list_id,
			'contact' => $contact_id,
			'status'  => $subscribe_status,
		);

		$response = $api_client->contacts()->updateListStatus( $args );
		$response->get();

		$this->maybe_log_errors( $response, $connection );
	}

	/**
	 * Remove a contact from a list.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $connection Connection data.
	 * @param array     $contact    Contact data.
	 */
	protected function api_contact_unsubscribe( $api_client, $connection, $contact ) {

		if (
			empty( $connection['list']['id'] ) ||
			! isset( $contact['contactLists'] )
		) {
			return;
		}

		$contact_lists = wp_list_pluck( $contact['contactLists'], 'status', 'list' );
		$contact_lists = array_map( 'absint', $contact_lists );

		// Prepare lists for unsubscribe.
		if ( 'all' === $connection['list']['id'] ) {
			$unsubscribe_lists = $contact_lists;
		} else {
			$_list_id          = absint( $connection['list']['id'] );
			$unsubscribe_lists = isset( $contact_lists[ $_list_id ] ) ? array( $_list_id => $contact_lists[ $_list_id ] ) : array();
		}

		// Update list(s) status for a contact.
		if ( ! empty( $unsubscribe_lists ) ) {
			$contact_endpoint = $api_client->contacts();

			foreach ( $unsubscribe_lists as $list_id => $status ) {

				// Skip if contact already unsubscribed from a list.
				if ( 2 === $status ) {
					continue;
				}

				// API call - unsubscribe a contact from a list.
				$response = $contact_endpoint->updateListStatus(
					array(
						'list'    => $list_id,
						'contact' => $contact['contact']['id'],
						'status'  => 2, // Unsubscribed value.
					)
				);
				$response->get();

				$this->maybe_log_errors( $response, $connection );
			}
		}
	}

	/**
	 * Sync contact tags.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $connection Connection data.
	 * @param int       $contact_id Contact ID.
	 * @param bool      $is_updated True if a contact was updated, false - if it was created new.
	 */
	protected function api_contact_tags_sync( $api_client, $connection, $contact_id, $is_updated ) {

		if (
			empty( $connection['tags']['add'] ) &&
			empty( $connection['tags']['delete'] )
		) {
			return;
		}

		$tags_contact = $this->api_contact_tags_get(
			$api_client,
			$connection,
			$contact_id
		);

		$tags_registered = $this->api_get_registered_tags(
			$api_client,
			array(
				'type' => 'contact',
			)
		);

		// Prepare tags.
		$tags_registered = $tags_registered ? wp_list_pluck( $tags_registered, 'tag', 'id' ) : array();
		$tags_contact    = $tags_contact ? array_map( 'absint', wp_list_pluck( $tags_contact, 'tag', 'id' ) ) : array();
		$tags_delete     = ! empty( $connection['tags']['delete'] ) ? array_map( 'sanitize_text_field', $connection['tags']['delete'] ) : array();
		$tags_add        = ! empty( $connection['tags']['add'] ) ? array_map( 'sanitize_text_field', $connection['tags']['add'] ) : array();

		// Clean tags that will be added - add only those that are not included in the delete tags array.
		$tags_add = array_diff( $tags_add, $tags_delete );

		$this->api_contact_tags_update(
			$api_client,
			$contact_id,
			array(
				'add'        => $tags_add,
				'delete'     => $tags_delete,
				'tagged'     => $tags_contact,
				'registered' => $tags_registered,
			)
		);
	}

	/**
	 * Retrieve contact tags.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $connection Connection data.
	 * @param int       $contact_id Contact ID.
	 *
	 * @return array
	 */
	protected function api_contact_tags_get( $api_client, $connection, $contact_id ) {

		// API call - retrieve contact tags.
		$r    = $api_client->contacts()->tags( $contact_id );
		$data = $r->get();

		// Request error.
		if ( $this->maybe_log_errors( $r, $connection ) ) {
			return array();
		}

		return $data['data']['contactTags'];
	}

	/**
	 * Update contact tags.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param int       $contact_id Contact ID.
	 * @param array     $tags       Multidimensional array with tag IDs.
	 */
	protected function api_contact_tags_update( $api_client, $contact_id, $tags ) {

		// API endpoint.
		$contact_endpoint = $api_client->contacts();

		// ADD tags.
		foreach ( $tags['add'] as $tag ) {
			$tag_id = $this->api_contact_get_tag_id_by_name( $api_client, $tag );

			if ( is_int( $tag_id ) ) {

				// API call - add a tag to contact.
				$contact_endpoint->tag( $contact_id, $tag_id );
			}
		}

		// DELETE tags.
		foreach ( $tags['delete'] as $tag ) {

			// Get registered tag ID.
			$registered_tag_id = array_search( $tag, $tags['registered'], true );

			if ( false === $registered_tag_id ) {
				continue;
			}

			// Get contactTag ID.
			$contact_tag_id = array_search( $registered_tag_id, $tags['tagged'], true );

			if ( false === $contact_tag_id ) {
				continue;
			}

			// API call - remove a tag from a contact.
			$contact_endpoint->untag( $contact_tag_id );
		}
	}

	/**
	 * Retrieve a tag ID by name.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param string    $tag        Tag name.
	 * @param bool      $create_new True if we need to create a new tag.
	 *
	 * @return int|boolean False on error.
	 */
	protected function api_contact_get_tag_id_by_name( $api_client, $tag, $create_new = true ) {

		$tags = $this->api_get_registered_tags(
			$api_client,
			array(
				'search' => $tag,
				'type'   => 'contact',
			)
		);

		// Tag already exists.
		if ( is_array( $tags ) && ! empty( $tags ) ) {
			$tags   = wp_list_pluck( $tags, 'tag', 'id' );
			$tag_id = array_search( $tag, $tags, true );

			if ( false !== $tag_id ) {
				return absint( $tag_id );
			}
		}

		// No need to create a tag if it does not exist.
		if ( ! $create_new ) {
			return false;
		}

		// API call - create a new tag.
		$response = $api_client->tags()->create(
			array(
				'tag' => $tag,
			)
		);

		$result = $response->get();

		if ( ! $result['success'] ) {
			return false;
		}

		return absint( $result['data']['tag']['id'] );
	}

	/**
	 * List all registered tags.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $args       Endpoint arguments.
	 *
	 * @return array|boolean False on error or if no tags.
	 */
	protected function api_get_registered_tags( $api_client, $args = array() ) {

		// API call - search a tag.
		$response = $api_client->tags()->listAll( $args );
		$result   = $response->get();

		if (
			! $result['success'] ||
			empty( $result['data']['tags'] )
		) {
			return false;
		}

		return $result['data']['tags'];
	}

	/**
	 * Create a note.
	 *
	 * @since 1.0.0
	 * @since 1.1.0 Allow smart tags in notes, allow to send note even when contact is updated.
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $connection Connection data.
	 * @param int       $contact_id Contact ID.
	 * @param bool      $is_updated True if a contact was updated, false - if was created new.
	 */
	protected function api_contact_note( $api_client, $connection, $contact_id, $is_updated ) {

		if ( empty( $connection['note'] ) ) {
			return;
		}

		$note = apply_filters( 'wpforms_process_smart_tags', $connection['note'], $this->form_data, $this->fields, $this->entry_id );

		// API call - create a Note.
		$api_client->notes()->create(
			array(
				'note'    => $note,
				'relid'   => $contact_id,
				'reltype' => 'Subscriber',
			)
		);
	}

	/**
	 * Create custom fields for a contact.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Client $api_client API client.
	 * @param array     $connection Connection data.
	 * @param int       $contact_id Contact ID.
	 * @param bool      $is_updated True if a contact was updated, false - if was created new.
	 */
	protected function api_contact_custom_fields( $api_client, $connection, $contact_id, $is_updated ) {

		if ( empty( $connection['fields_meta'] ) ) {
			return;
		}

		// API call - retrieve all custom fields filtered by list ID.
		$args = array(
			'listfilter' => ! empty( $connection['list']['id'] ) ? absint( $connection['list']['id'] ) : '',
		);

		$response         = $api_client->contacts()->listAllCustomFields( $args );
		$ac_custom_fields = $response->get();

		// Request error.
		if ( $this->maybe_log_errors( $response, $connection ) ) {
			return;
		}

		$contact_data = array();
		if ( $is_updated ) {

			// API call - retrieve full data about contact.
			$r             = $api_client->contacts()->get( $contact_id );
			$_contact_data = $r->get();

			if ( ! $this->maybe_log_errors( $r, $connection ) ) {
				$contact_data = $_contact_data['data'];
			}
		}

		$contact_custom_fields  = isset( $contact_data['fieldValues'] ) ? wp_list_pluck( $contact_data['fieldValues'], 'value', 'field' ) : array();
		$ac_custom_fields       = wp_list_pluck( $ac_custom_fields['data']['fields'], 'type', 'id' );
		$ac_custom_field_values = $this->prepare_custom_field_values( $connection['fields_meta'], $ac_custom_fields, $contact_custom_fields );

		$client_endpoint = $api_client->contacts();
		foreach ( $ac_custom_field_values as $item ) {

			// API call - create a custom field value.
			$client_endpoint->createCustomFieldValue( $contact_id, $item['field_id'], $item['field_value'] );
		}
	}

	/**
	 * Retrieve Custom Field values for mapping with ActiveCampaign.
	 *
	 * @since 1.0.0
	 *
	 * @param array $fields_meta           Connection fields meta.
	 * @param array $all_custom_fields     All custom fields from ActiveCampaign filtered by list ID.
	 * @param array $contact_custom_fields Already saved custom field values for current contact.
	 *
	 * @return array
	 */
	protected function prepare_custom_field_values( $fields_meta, $all_custom_fields, $contact_custom_fields ) {

		$result = array();

		foreach ( $fields_meta as $item ) {

			if (
				empty( $item['name'] ) ||
				( empty( $item['field_id'] ) && 0 !== $item['field_id'] ) ||
				empty( $this->fields[ $item['field_id'] ] )
			) {
				continue;
			}

			$ac_field_id = $item['name'];

			if ( ! isset( $all_custom_fields[ $ac_field_id ] ) ) {
				continue;
			}

			$ac_field_value = $this->format_custom_field_value( $item['field_id'], $all_custom_fields[ $ac_field_id ] );

			if (
				! empty( $contact_custom_fields ) &&
				(
					// If current value was set and equal with new value.
					(
						isset( $contact_custom_fields[ $ac_field_id ] ) &&
						$contact_custom_fields[ $ac_field_id ] === $ac_field_value
					) ||

					// If current value not set before (or empty) and new field value is empty too.
					(
						empty( $contact_custom_fields[ $ac_field_id ] ) &&
						empty( $ac_field_value )
					)
				)
			) {
				continue;
			}

			$result[] = array(
				'field_id'    => $ac_field_id,
				'field_value' => $ac_field_value,
			);
		}

		return $result;
	}

	/**
	 * Apply special formats for ActiveCampaign fields.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $field_id      Form field ID.
	 * @param string $ac_field_type ActiveCampaign field type.
	 *
	 * @return mixed
	 */
	protected function format_custom_field_value( $field_id, $ac_field_type ) {

		$field_value    = $this->get_form_field_value( $field_id );
		$ac_field_value = $field_value;

		switch ( $ac_field_type ) {
			case 'checkbox':
			case 'listbox':
				if ( ! empty( $field_value ) ) {
					$ac_field_value = str_replace( "\n", '||', $field_value );
					$ac_field_value = sprintf( '||%s||', $ac_field_value );
				}
				break;

			case 'date':
				$ac_field_value = $this->format_date( $this->form_data['fields'][ $field_id ], $this->fields[ $field_id ], 'Y-m-d' );
				break;
		}

		/**
		 * Filter a value for ActiveCampaign custom field.
		 *
		 * @since 1.0.0
		 *
		 * @param string|int $ac_field_value Formatted field value.
		 * @param string     $ac_field_type  ActiveCampaign field type.
		 * @param int        $field_id       Form field ID.
		 * @param array      $data           All data about form submission.
		 */
		return apply_filters(
			'wpforms_activecampaign_format_custom_field_value',
			$ac_field_value,
			$ac_field_type,
			$field_id,
			array(
				'fields'    => $this->fields,
				'entry'     => $this->entry,
				'form_data' => $this->form_data,
				'entry_id'  => $this->entry_id,
			)
		);
	}

	/**
	 * Retrieve a form field value.
	 *
	 * @since 1.0.0
	 *
	 * @param int $field_id The Field ID.
	 *
	 * @return string
	 */
	protected function get_form_field_value( $field_id ) {

		// Submitted field value.
		$field_value = $this->fields[ $field_id ]['value'];

		// If payment type fields.
		if ( in_array( $this->fields[ $field_id ]['type'], [ 'payment-checkbox', 'payment-multiple', 'payment-select' ], true ) ) {

			// Make a delimiter like in `Checkbox` field.
			$field_value = str_replace( "\r\n", "\n", $this->fields[ $field_id ]['value_choice'] );
		}

		if ( in_array( $this->fields[ $field_id ]['type'], [ 'payment-single', 'payment-total' ], true ) ) {

			// Additional conversion for correct currency symbol display.
			$field_value = wpforms_decode_string( $field_value );
		}

		return $field_value;
	}

	/**
	 * Convert a date value into an expected format.
	 *
	 * @since 1.0.0
	 *
	 * @param array  $field_data      Field data.
	 * @param array  $field           Field attributes.
	 * @param string $expected_format Date format.
	 *
	 * @return string
	 */
	protected function format_date( $field_data, $field, $expected_format ) {

		$result = '';

		if (
			empty( $field_data['format'] ) ||
			! in_array( $field_data['format'], array( 'date', 'date-time' ), true )
		) {
			return $result;
		}

		// Parse a value with date string according to a specified format.
		$date_time = false;
		if ( ! empty( $field_data['date_format'] ) ) {
			$date_time = date_create_from_format( $field_data['date_format'], $field['value'] );
		}

		// Fallback with using timestamp value.
		if ( ! $date_time && ! empty( $field['unix'] ) ) {
			$date_time = date_create( '@' . $field['unix'] );
		}

		if ( $date_time ) {
			$result = $date_time->format( $expected_format );
		}

		return $result;
	}

	/**
	 * Check if API response has errors and log them if any.
	 *
	 * @since 1.0.0
	 *
	 * @param AC_Response $response   An object with response.
	 * @param array       $connection Specific connection data that errored.
	 *
	 * @return bool True when there is something to log (we have an error), false otherwise.
	 */
	protected function maybe_log_errors( $response, $connection ) {

		$data = $response->get_output();

		if ( ! $data['success'] ) {
			$this->log_errors( $response->get_input(), $connection, $data['message'] );

			return true;
		}

		return false;
	}

	/**
	 * Apply delay execution.
	 *
	 * @since 1.0.0
	 */
	protected function apply_sleep() {

		sleep( 2 );
	}

	/**
	 * Log an API-related error with all the data.
	 *
	 * @since 1.0.0
	 *
	 * @param mixed  $response   Response data, may be array or \WP_Error.
	 * @param array  $connection Specific connection data that errored.
	 * @param string $message    Error message.
	 */
	protected function log_errors( $response, $connection, $message ) {

		wpforms_log(
			esc_html__( 'Submission to ActiveCampaign failed.', 'wpforms-activecampaign' ) . "(#{$this->entry_id})",
			array(
				'response'       => $response,
				'connection'     => $connection,
				'activecampaign' => $message,
			),
			array(
				'type'    => array( 'provider', 'error' ),
				'parent'  => $this->entry_id,
				'form_id' => $this->form_data['id'],
			)
		);
	}
}
