<?php
/**
 * Plugin Name:       WPForms ActiveCampaign
 * Plugin URI:        https://wpforms.com
 * Description:       ActiveCampaign integration with WPForms.
 * Requires at least: 5.2
 * Requires PHP:      5.6
 * Author:            WPForms
 * Author URI:        https://wpforms.com
 * Version:           1.5.0
 * Text Domain:       wpforms-activecampaign
 * Domain Path:       languages
 *
 * WPForms is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * any later version.
 *
 * WPForms is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with WPForms. If not, see <https://www.gnu.org/licenses/>.
 *
 * @since     1.0.0
 * @author    WPForms
 * @package   WPFormsActiveCampaign
 * @license   GPL-2.0+
 * @copyright Copyright (c) 2020, WPForms LLC
 */

use WPFormsActiveCampaign\Plugin;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin constants.
 *
 * @since 1.0.0
 */
// phpcs:disable WPForms.Comments.PHPDocDefine.MissPHPDoc
define( 'WPFORMS_ACTIVECAMPAIGN_VERSION', '1.5.0' );
define( 'WPFORMS_ACTIVECAMPAIGN_FILE', __FILE__ );
define( 'WPFORMS_ACTIVECAMPAIGN_PATH', plugin_dir_path( WPFORMS_ACTIVECAMPAIGN_FILE ) );
define( 'WPFORMS_ACTIVECAMPAIGN_URL', plugin_dir_url( WPFORMS_ACTIVECAMPAIGN_FILE ) );
// phpcs:enable WPForms.Comments.PHPDocDefine.MissPHPDoc

/**
 * Check addon requirements.
 * We do it on `wpforms_loaded` hook. If earlier - core is still not loaded.
 *
 * @since 1.0.0
 * @since 1.2.0 Changed autoloader - started using composer things.
 */
function wpforms_activecampaign_required() {

	if ( PHP_VERSION_ID < 50600 ) {
		add_action( 'admin_init', 'wpforms_activecampaign_deactivation' );
		add_action( 'admin_notices', 'wpforms_activecampaign_fail_php_version' );

	} elseif (
		! function_exists( 'wpforms' ) ||
		version_compare( wpforms()->version, '1.7.5', '<' )
	) {
		add_action( 'admin_init', 'wpforms_activecampaign_deactivation' );
		add_action( 'admin_notices', 'wpforms_activecampaign_fail_wpforms_version' );

	} elseif (
		! function_exists( 'wpforms_get_license_type' ) ||
		! in_array( wpforms_get_license_type(), [ 'elite', 'agency', 'ultimate' ], true )
	) {
		return;

	} else {
		// Actually, load the ActiveCampaign addon now.
		require_once WPFORMS_ACTIVECAMPAIGN_PATH . 'vendor/autoload.php';
		Plugin::get_instance();
	}
}

add_action( 'wpforms_loaded', 'wpforms_activecampaign_required' );

/**
 * Deactivate the plugin.
 *
 * @since 1.0.0
 */
function wpforms_activecampaign_deactivation() {

	deactivate_plugins( plugin_basename( __FILE__ ) );
}

/**
 * Admin notice for a minimum PHP version.
 *
 * @since 1.0.0
 */
function wpforms_activecampaign_fail_php_version() {

	echo '<div class="notice notice-error"><p>';
	printf(
		wp_kses( /* translators: %s - WPForms.com documentation page URI. */
			__( 'The WPForms ActiveCampaign plugin has been deactivated. Your site is running an outdated version of PHP that is no longer supported and is not compatible with the ActiveCampaign plugin. <a href="%s" target="_blank" rel="noopener noreferrer">Read more</a> for additional information.', 'wpforms-activecampaign' ),
			[
				'a' => [
					'href'   => [],
					'rel'    => [],
					'target' => [],
				],
			]
		),
		'https://wpforms.com/docs/supported-php-version/'
	);
	echo '</p></div>';

	// phpcs:disable WordPress.Security.NonceVerification.Recommended
	if ( isset( $_GET['activate'] ) ) {
		unset( $_GET['activate'] );
	}
	// phpcs:enable WordPress.Security.NonceVerification.Recommended
}

/**
 * Admin notice for minimum WPForms version.
 *
 * @since 1.0.0
 */
function wpforms_activecampaign_fail_wpforms_version() {

	echo '<div class="notice notice-error"><p>';
	esc_html_e( 'The WPForms ActiveCampaign plugin has been deactivated, because it requires WPForms v1.7.5 or later to work.', 'wpforms-activecampaign' );
	echo '</p></div>';

	// phpcs:disable WordPress.Security.NonceVerification.Recommended
	if ( isset( $_GET['activate'] ) ) {
		unset( $_GET['activate'] );
	}
	// phpcs:enable WordPress.Security.NonceVerification.Recommended
}
